using System;
using System.Data;
using gov.va.med.vbecs.ExceptionManagement;
using System.Reflection;
using System.Text.RegularExpressions;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;
using SPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using System.Collections;
using BROKENRULES = gov.va.med.vbecs.Common.VbecsBrokenRules.FrmModifyUnits;

namespace gov.va.med.vbecs.BOL
{

	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>John Vrooland</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>3/12/2003</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>Implements UC_24,25,26</summary>
	///
	#endregion


	public class BloodUnitModification : BaseBusinessObject
	{
		#region Variables

		private bool _originalHasB2Set = false;

		/// <summary>
		/// BR_26.25 - Carry over any patient assignments (as selected by user)
		/// </summary>
		private System.Guid[] _inheritedPatientAssignment = {System.Guid.Empty,System.Guid.Empty};

		private System.Collections.ArrayList _poolingUnits;
		private BOL.ModifiedUnit _pooledUnit;

		private System.Collections.ArrayList _splitUnits;
		private BOL.BloodUnit _splittingUnit;

		private BOL.BloodUnit _originalUnit;
		private BOL.ModifiedUnit _modifiedUnit;

		// Need class level instances of these because we can modify in batch
		private System.Collections.ArrayList _labelExceptions = new ArrayList();
		private System.Collections.ArrayList _exceptions = new ArrayList();

		//Stores the UnitId of the Pool, so we know to check it at save time and either update/leave the PoolCounter record
		private string _calculatedPooledUnitId = string.Empty;
		private System.Data.DataRow _drUsedPoolCounter;		//Save off the PoolCounter for update in Save method (PrepareAndSavePool)  -CR3320

		private Common.ProductModification _pooledProductModificationCode;
		private Common.ProductModification _productModificationCode;
		private Common.ModifyUnitMethod _modificationMethod;
		private System.DateTime _modificationDateTime;
		private string _modificationTechID;

		private decimal _modificationCost = decimal.MinusOne;
		private System.Data.DataTable _dtExpirationRules = null;

		/// <summary>
		/// DataTable of the original units contained in a pool - for Edit's only
		/// </summary>
		private System.Data.DataTable _dtOriginalPoolingUnits;
		private System.Data.DataTable _dtOriginalPoolEquipment;
		private System.Data.DataTable _dtOriginalPoolSupply;

		private bool _newContainerWeldIndicator = false;

//		//Used when calculating ExpirationDates
//		/// <summary>
//		/// # of hours to add to Modification Date for Closed Modification - based on TT_26.02
//		/// CR2002 - This value needs to be available when Editing a Pool, so made public for ModifiedUnit to access
//		/// </summary>
//		private int _closedExpirationDateRuleValue = 0;
//		/// <summary>
//		/// # of hours to add to Modification Date for Open Modification - based on TT_26.02
//		/// CR2002 - This value needs to be available when Editing a Pool, so made public for ModifiedUnit to access
//		/// </summary>
//		private int _openExpirationDateRuleValue = 0;

		/// <summary>
		/// Keeps track of the original units that made up a pool mod, so that we can stop the user
		/// from removing ALL of the original units during an Edit Pool -- BR_24.28
		/// </summary>
		private System.Collections.ArrayList _originalPoolingUnits = new System.Collections.ArrayList();
		private System.Collections.ArrayList _remainingOriginalUnits = new System.Collections.ArrayList();

		private Common.UpdateFunction _lastUpdateFunctionId;

		private string _formName, _ruleSet;
		//For PoolEdits -- the guid of the "pooled" unit that's being re-created
		private System.Guid _originalBloodUnitGuid = System.Guid.Empty;

		#endregion

		#region Constructors

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="159"> 
		///		<ExpectedInput>ProductModificationCode, ModificationMethod, ModificationDateTime, TechId</ExpectedInput>
		///		<ExpectedOutput>BloodUnitModification object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="160"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///  Constructor for this control class
		/// </summary>
		/// <param name="productModification">Type of modification being performed</param>
		/// <param name="modificationMethod">Open, Closed, SCD</param>
		/// <param name="modificationDateTime">Time of the modification</param>
		/// <param name="modificationTechID">Tech preforming the modification</param>
		public BloodUnitModification(Common.ProductModification productModification,
			Common.ModifyUnitMethod modificationMethod,
			System.DateTime modificationDateTime,
			string modificationTechID)
		{
			this._poolingUnits = new System.Collections.ArrayList();
			this._splitUnits = new System.Collections.ArrayList();

			//Save the modify variables
			this._modificationMethod = modificationMethod;
			this._modificationDateTime = modificationDateTime;
			this._productModificationCode = productModification;
			this._pooledProductModificationCode = productModification;	//Saving this for FrmPoolUnits - GetTargetProducts
			this._modificationTechID = modificationTechID;
		}

		#endregion

		#region Events

		/// <summary>
		///	Lets the UI listen to parent/child changes
		/// </summary>
		public event EventHandler ModifiedUnitStateChange;

		/// <summary>
		/// Allows this control to filter all child units through one event for valid/invalid changes
		/// </summary>
		protected void OnModifiedUnitStateChange()
		{
			if (ModifiedUnitStateChange != null)
				ModifiedUnitStateChange(this, System.EventArgs.Empty);
		}

		/// <summary>
		/// This is the event we attach to our children's valid/invalid events
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		protected void HandleModifiedUnitStateChange(object sender, System.EventArgs e)
		{
			this.OnModifiedUnitStateChange();
		}

		/// <summary>
		///	Lets the UI listen to parent/child changes
		/// </summary>
		public event EventHandler WeldIndicatorChanged;

		/// <summary>
		/// Allows this control to filter all child units through one event for valid/invalid changes
		/// </summary>
		protected void OnWeldIndicatorChange()
		{
			if (WeldIndicatorChanged != null)
				WeldIndicatorChanged(this, System.EventArgs.Empty);
		}

		/// <summary>
		/// Lets the UI listen to parent volume changes (since we do it for the user)
		/// </summary>
		public event EventHandler ParentVolumeRefresh;

		/// <summary>
		/// This event will raise back to the UI to update the Splitting Unit's volume
		/// </summary>
		protected void OnParentVolumeRefresh()
		{
			if (ParentVolumeRefresh != null)
				ParentVolumeRefresh(this, System.EventArgs.Empty);
		}

		/// <summary>
		/// Recalculate the volumes after a user change to make sure no rules were broken
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		protected void PoolUnitVolumeChanged(object sender, System.EventArgs e)
		{
			decimal totalChildVolume = this.TotalVolumeOfPoolingUnits;

			//If they've made the split unit volumes too high, then set a brokenrule -- BR_
			if (this.PooledUnit.NewUnit.OriginalVolume < 0 || this.PooledUnit.NewUnit.OriginalVolume > 2000)
				this.RuleBroken(BROKENRULES.Volume, true);
			else
				this.RuleBroken(BROKENRULES.Volume, false);
		}

		/// <summary>
		/// Recalculate the volumes after a user change to make sure no rules were broken
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		protected void ModifyUnitVolumeChanged(object sender, System.EventArgs e)
		{
			if (this.ProductModificationCode == Common.ProductModification.VolumeReduced)
			{
				//Changed from originalUnit.Discard....  BR_26.41 change
				decimal totalVolume = this._modifiedUnit.NewUnit.OriginalVolume + this._modifiedUnit.NewUnit.DiscardedPlasmaVolume;
				//Set the DiscardVolume broken Rule for VolumeReduction modifications because it's more 
				//descriptive for the user
				this.RuleBroken(BROKENRULES.DiscardVolume , totalVolume > this._originalUnit.OriginalVolume);
			}
			else
			{
				this.RuleBroken(BROKENRULES.Volume, this._modifiedUnit.NewUnit.OriginalVolume > this._originalUnit.OriginalVolume);
			}
		}

		/// <summary>
		/// Recalculate the volumes after a user change to make sure no rules were broken
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		protected void SplitUnitVolumeChanged(object sender, System.EventArgs e)
		{
			BOL.ModifiedUnit childUnit = (BOL.ModifiedUnit)sender;

			bool ruleBroken = (childUnit.NewUnit.OriginalVolume > 2000 || childUnit.NewUnit.OriginalVolume < 1);
			
			this.RuleBroken(BROKENRULES.Volume, ruleBroken);

			this.OnParentVolumeRefresh();
		}

		/// <summary>
		/// Total volume of all the units in pool
		/// </summary>
		private decimal TotalVolumeOfPoolingUnits
		{
			get
			{
				decimal totalChildVolume = decimal.Zero;

				foreach(BOL.BloodUnit tmpPoolUnit in this._poolingUnits)
				{
					totalChildVolume += tmpPoolUnit.OriginalVolume;
				}

				return(totalChildVolume);
			}
		}

		#endregion

		#region Properties

	
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6975"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>decimal</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6976"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Removed BR 25.09 - which caused a need for the set{}...
		/// </summary>
		public decimal OriginalUnitVolume
		{
			get
			{
				if (this.IsSplit)
				{
					return(this.SplittingUnit.OriginalVolume);
				}
				else
				{
					return (this.OriginalUnit.OriginalVolume);
				}
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/25/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6489"> 
		///		<ExpectedInput>Guid Array</ExpectedInput>
		///		<ExpectedOutput>Guid Array</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6490"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Arraylist of BloodUnit/PatientGuids of assignments to be saved to the modified unit
		/// </summary>
		public System.Guid[] InheritedPatientAssignment
		{
			get
			{				
				return(this._inheritedPatientAssignment);
			}
			set
			{
				this._inheritedPatientAssignment = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6634"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>boolean</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6635"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns True if they do!
		/// </summary>
		public bool DoPoolingUnitsHaveAssignedPatients
		{
			get
			{
				int patientCount = 0;

				foreach(BOL.BloodUnit tmpUnit in this._poolingUnits)
				{
					patientCount += tmpUnit.AssignedToPatients.Count;
				}

				return (patientCount > 0);
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="161"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>BloodUnit</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="162"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// When performing a non Pool/Split, this will be the starting unit before modification occurs
		/// </summary>
		public BOL.BloodUnit OriginalUnit
		{
			get
			{
				return this._originalUnit;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="163"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="164"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CMV Negative status of the target unit (calculated based on pooling-units for BR_24.19)
		/// </summary>
		public bool TargetCmvNeg
		{
			get
			{
				bool targetValue = true;

				foreach(BOL.BloodUnit pooledUnit in this._poolingUnits)
				{
					//If only one is false, then the target pool is false
					if (!pooledUnit.CmvNegative) 
					{
						targetValue = false;
						break;
					}
				}
				return(targetValue);
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="165"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="166"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns unit in the collection using the original container
		/// </summary>
		public int UnitUsingOriginalContainer
		{
			get
			{
				for(int ndx=0; ndx < this.SplitCount; ndx++)
				{
					if (((BOL.ModifiedUnit)this._splitUnits[ndx]).OriginalContainerIndicator)
					{
						return ndx;
					}
				}
				return int.MinValue;
			}
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="167"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>ModifiedUnit</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="168"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// When performing a non Pool/Split, this will be the modified unit, replacing the original unit
		/// </summary>
		public BOL.ModifiedUnit ModifiedUnit
		{
			get
			{
				return this._modifiedUnit;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6977"> 
		///		<ExpectedInput>ArrayList</ExpectedInput>
		///		<ExpectedOutput>ArrayList</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6978"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Exceptions
		/// </summary>
		public System.Collections.ArrayList Exceptions
		{
			get
			{
				return(this._exceptions);
			}
			set
			{
				this._exceptions = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6979"> 
		///		<ExpectedInput>ArrayList</ExpectedInput>
		///		<ExpectedOutput>ArrayList</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6980"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Label Exceptions
		/// </summary>
		public System.Collections.ArrayList LabelExceptions
		{
			get
			{
				return(this._labelExceptions);
			}
			set
			{
				this._labelExceptions = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="129"> 
		///		<ExpectedInput>bool</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="130"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Weld Indicator field - affects the unit's expriation date
		/// </summary>
		public bool NewContainerWeldIndicator
		{
			get
			{
				return this._newContainerWeldIndicator;
			}
			set
			{
				this._newContainerWeldIndicator = value;

				if (this.IsSplit)
				{
					foreach(BOL.ModifiedUnit tmpSplit in this._splitUnits)
					{
						tmpSplit.WeldIndicator = value;
						tmpSplit.UnitExpirationDate = tmpSplit.RecommendedExpirationDateTime(this);
					}
				}
				else if (this.IsPool)
				{
					this.PooledUnit.WeldIndicator = value;
					this.PooledUnit.UnitExpirationDate = this.PooledUnit.RecommendedExpirationDateTime(this);
				}
				else
				{
						this.ModifiedUnit.WeldIndicator = value;
						this.ModifiedUnit.UnitExpirationDate = this.ModifiedUnit.RecommendedExpirationDateTime(this);
				}
				this.RuleBroken(MethodBase.GetCurrentMethod().Name, false);
				this.OnWeldIndicatorChange();
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="169"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>ArrayList</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="170"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Collection of BOL.BloodUnits that are being pooled into one
		/// </summary>
		public System.Collections.ArrayList PoolingUnits
		{
			get
			{
				return this._poolingUnits;
			}
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="171"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>ModifiedUnit</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="172"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// The target unit that all the poolingunits are joined to form
		/// </summary>
		public BOL.ModifiedUnit PooledUnit
		{
			get
			{
				return this._pooledUnit;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="173"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>BloodUnit</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="174"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// The bloodunit that is being split
		/// </summary>
		public BOL.BloodUnit SplittingUnit
		{
			get
			{
				return this._splittingUnit;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="175"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>ArrayList</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="176"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Collection of BOL.ModifiedUnits that were formed as a result of the split
		/// </summary>
		public System.Collections.ArrayList SplitUnits
		{
			get
			{
				return this._splitUnits;
			}
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="177"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="178"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Number of split units
		/// </summary>
		public int SplitCount
		{
			get
			{
				return this._splitUnits.Count;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="179"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="180"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// BR_24.23
		/// </summary>
		public int MinimumPooledUnitsAllowed
		{
			get
			{
				return (int)2;
			}
		}
		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="181"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="182"> 
		///		<ExpectedInput>Invalid OriginalVolume</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// BR_25.01 -- 2 to 26 splits are allowed, or until the volume of the original unit is consumed
		/// </summary>
		public int MaximumSplitUnitsAllowed
		{
			get
			{
				decimal maxOrigSplits = this.SplittingUnit.OriginalVolume;

				if (maxOrigSplits < 2)
					throw new BOL.BusinessObjectException();

				//Accoring to the BR, 26 is the max number of splits, but if they don't have enough volume, then
				//the new MaxSplits is the volume#
				if (maxOrigSplits < 26)
					return (int)maxOrigSplits;
				else
					return (int)26;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="183"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="184"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// BR 25.01 -- 2 to 26 splits are allowed
		/// </summary>
		public static int DefaultSplitUnits
		{
			get
			{
				return (int)2;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="185"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DateTime</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="186"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// BR_26.01 - Modifications cannot be recorded with FUTURE dates
		/// </summary>
		public static System.DateTime MaximumModificationDateTime
		{
			get
			{
				return VBECSDateTime.GetDivisionCurrentDateTime();
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="187"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Common.ProductModification</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="188"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Modification Type
		/// </summary>
		public Common.ProductModification ProductModificationCode
		{
			get
			{
				return this._productModificationCode;
			}
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6981"> 
		///		<ExpectedInput>Common.ProductModification</ExpectedInput>
		///		<ExpectedOutput>Common.ProductModification</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6982"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// The original ProductModificationCode of a pooled unit.
		/// Since edits have the ProductModification code EditPool, we need to know what type
		/// of pool we're editing to apply the correct BRs.
		/// </summary>
		public Common.ProductModification ProductModificationCodeBase
		{
			get
			{
				if (this.ProductModificationCode == Common.ProductModification.EditPool)
				{
					return this._pooledProductModificationCode;
				}
				else
				{
					return this.ProductModificationCode;
				}
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="189"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Common.ModifyUnitMethod</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="190"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Medification Method.  BR_26.13
		/// </summary>
		public Common.ModifyUnitMethod ModificationMethod
		{
			get
			{
				return this._modificationMethod;
			}
		}

		/// <summary>
		/// Grabs the cost assigned in UC_9 from the ProductModificationParameter table
		/// </summary>
		private decimal ModificationCost
		{
			get
			{
				if (this._modificationCost == decimal.MinusOne)
				{
					BOL.BloodProductModifications tmpBPM = new BOL.BloodProductModifications(Common.LogonUser.LogonUserDivisionCode);
					this._modificationCost = (decimal)tmpBPM.ModificationCosts[Common.Utility.GetProductModificationCharFromEnum(this.ProductModificationCodeBase)];
				}

				return(this._modificationCost);
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="191"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DateTime</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="192"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Modification DateTime, implements 
		/// </summary>
		public System.DateTime ModificationDateTime
		{
			get
			{
				return this._modificationDateTime;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="119"> 
		///		<ExpectedInput>Valid String</ExpectedInput>
		///		<ExpectedOutput>Valid String</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="120"> 
		///		<ExpectedInput>Empty String</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// User performing the modification
		/// </summary>
		public string ModificationTechId
		{
			get
			{
				return this._modificationTechID;
			}
			set
			{
				if (value == string.Empty)
				{
					throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument(MethodBase.GetCurrentMethod().Name).ResString);
				}
				this._modificationTechID = value;
				this.IsDirty = true;
			}
		}


		#endregion

		#region Methods

	    ///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7389"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7390"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// InactivatePoolCounter
		/// </summary>
		/// <returns></returns>
		public bool InactivatePoolCounter()
		{
            //CR3320 - reworked to use _drUsedPoolCounter, instead of DataTable
            bool retValue = true;

			//If _calculatedPoolUnitId is blank, they dont have a counter to inactivate -- could be codabar if it's different since codabar alllows manual pool ID entry
            if (this._calculatedPooledUnitId == string.Empty || this._calculatedPooledUnitId != this.PooledUnit.NewUnit.EyeReadableUnitId) return (retValue);

			//If the PoolCounter table has no rows in it, it means the table was blank when the user tried this..
            if (this._drUsedPoolCounter != null)
			{
                retValue = DAL.BloodUnitModification.InactivatePoolCounter(this._drUsedPoolCounter);
				this._calculatedPooledUnitId = string.Empty;        //reset
			}
			return(retValue);
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7391"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7392"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CalculateNewPoolId
		/// </summary>
        /// <param name="lockController">LOCKING2.0 - Controller handles locking</param>
		/// <returns></returns>
		public string GetNewPoolId(BOL.LockManager lockController)
		{
            //CR3320 - fixed code after GetNextAvailablePoolId was added (recursive method)
            string modYear = this.ModificationDateTime.ToString("yy");
			string unitID = string.Empty;

		    DataTable dtCounters = DAL.BloodUnitModification.GetPoolCounter(modYear);
		    int counterValue = this.GetNextAvaiablePoolId(dtCounters, lockController, 0, 0, modYear);

			//Assign the EyeReadable
			switch (this._pooledUnit.NewUnit.BarcodeType)
			{
					//BR_24.11
				case Common.BarcodeType.Codabar:
					
					unitID = this._pooledUnit.NewUnit.DivisionCode.Replace(" ", "0") + modYear + counterValue.ToString();
					break;

					//BR_24.12
				case Common.BarcodeType.ISBT128:

					BOL.Division div = new BOL.Division(Common.LogonUser.LogonUserDivisionCode);
					unitID = div.IccbbaRegistrationNumber + modYear + counterValue.ToString("000000");

					this._pooledUnit.NewUnit.UnitDivisionId = "00";

					break;
			}

			this._calculatedPooledUnitId = unitID;

			return(unitID);
		}


        /// <summary>
        /// New recursive method to get the available pool id CR3320
        /// </summary>
        /// <param name="dtCounters">DataTable of "A" and top "X" PoolConter rows</param>
        /// <param name="lockController">For Locking, passed in from calling form</param>
        /// <param name="biggestX">highest CounterValue found when RecordStatusCode = 'X'</param>
        /// <param name="biggestA">highest CounterValue found when RecordStatusCode = 'A'</param>
        /// <param name="modYear">year of the modification</param>
        /// <returns></returns>
        private int GetNextAvaiablePoolId(System.Data.DataTable dtCounters, BOL.LockManager lockController, int biggestX, int biggestA, string modYear)
        {
            int newCounterValue = (biggestX > biggestA) ? biggestX : biggestA;         //Get the higher number
            int currentA = 0;
            int currentX = 0;
            newCounterValue++;  //add one

            //If there's no rows, then there's No A's or X's
            if (dtCounters.Rows.Count == 0)
            {
                System.Guid newpoolCounterGuid = Guid.NewGuid();
                lockController.LockRecord(newpoolCounterGuid, false); //Lock the new record before we create it to prevent snatches
                this._drUsedPoolCounter = DAL.BloodUnitModification.InsertPoolCounter(newpoolCounterGuid, newCounterValue, modYear);
                return (newCounterValue);
            }
            else
            {
                DataRow[] openCounters = dtCounters.Select(TABLE.PoolCounter.RecordStatusCode + " = 'A'", TABLE.PoolCounter.CounterValue);        //ASC to fill in the smaller #s first
                DataRow[] closedCounters = dtCounters.Select(TABLE.PoolCounter.RecordStatusCode + " = 'X'", TABLE.PoolCounter.CounterValue + " DESC");

                //Get the CounterValues and save the biggest
                if (openCounters.Length > 0) currentA = System.Convert.ToInt32(openCounters[0][TABLE.PoolCounter.CounterValue].ToString());
                if (closedCounters.Length > 0) currentX = System.Convert.ToInt32(closedCounters[0][TABLE.PoolCounter.CounterValue].ToString());
                biggestX = (currentX > biggestX) ? currentX : biggestX;
                biggestA = (currentA > biggestA) ? currentA : biggestA;
                newCounterValue = (biggestX > biggestA) ? biggestX : biggestA;
                newCounterValue++;      //add one since it will be new

                if (openCounters.Length == 0) //dtCounters only has "X" rows
                {
                    System.Guid newpoolCounterGuid = Guid.NewGuid();
                    lockController.LockRecord(newpoolCounterGuid, false); //Lock the new record before we create it to prevent snatches
                    this._drUsedPoolCounter = DAL.BloodUnitModification.InsertPoolCounter(newpoolCounterGuid, newCounterValue, modYear);
                    return (newCounterValue);
                }
                else //We have "A"s
                {
                    if (openCounters[0].IsNull(TABLE.VbecsLock.VbecsLockGuid))      //The record is not currently locked
                    {
                        System.Guid exitingCounterGuid = (System.Guid)openCounters[0][TABLE.PoolCounter.PoolCounterGuid];
                        BOL.Lock tmpLock = lockController.LockRecord(exitingCounterGuid, false); //Lock the new record before we create it to prevent snatches
                        if (tmpLock.LockConflictIndicator(lockController.LockedFormId))
                        {
                            dtCounters.Rows.Remove(openCounters[0]);
                            return (this.GetNextAvaiablePoolId(dtCounters, lockController, biggestX, biggestA, modYear));
                        }
                        else
                        {
                            this._drUsedPoolCounter = openCounters[0];
                            return (currentA);
                        }
                    }
                    else
                    {
                        //The record is currently locked, so lets recurse
                        dtCounters.Rows.Remove(openCounters[0]);
                        return (this.GetNextAvaiablePoolId(dtCounters, lockController, biggestX, biggestA, modYear));
                    }
                }
            }

        }

	    ///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6983"> 
		///		<ExpectedInput>Unit Guid</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6984"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// PoolEdits ONLY
		/// </summary>
		/// <param name="pooledUnitGuid"></param>
		public void EditPool(System.Guid pooledUnitGuid)
		{
			this._poolingUnits = new System.Collections.ArrayList();

			//This is probably a lot more complicated than what I have here, but it's a start...
			//This basically returns the BLoodUnitModification table for our previous pool transaction
			this._dtOriginalPoolingUnits = DAL.BloodUnitModification.GetBloodUnitsInPool(pooledUnitGuid);

			//PULL IN THE ORIGINAL DATE -- PER KAREN - CUZ THEIR EDITING A UNIT
			this._modificationDateTime = (System.DateTime) _dtOriginalPoolingUnits.Rows[0][TABLE.BloodUnitModification.ModificationDate];
			this._modificationMethod = Common.Utility.GetModificationMethodEnumFromString(this._dtOriginalPoolingUnits.Rows[0][TABLE.BloodUnitModification.ModificationMethodCode].ToString());
			this._productModificationCode = Common.ProductModification.EditPool;

			//We can use the zero row, beacause ModifiedUnit(dtRow) looks at the ToBloodUnitGuid to load
			this._pooledUnit = new BOL.ModifiedUnit(this._dtOriginalPoolingUnits.Rows[0]);

			//Doing this to grab the UnitExpirationDate of the Existing Pool
			BOL.BloodUnitMedia tmpMedia = new BOL.BloodUnitMedia(pooledUnitGuid);
			this._pooledUnit.UnitExpirationDate = tmpMedia.UnitExpirationDate;
			
			//Set this on the BloodUnitModification controller so that it can be accessed for all units
			//Needed because a Edit Pool mod, can be originally a Pool, or Pool/Thaw mod
			this._pooledProductModificationCode = this._pooledUnit.ProductModificationCode;

			foreach(System.Data.DataRow dr in this._dtOriginalPoolingUnits.Rows)
			{
				System.Guid fromGuid = (System.Guid)dr[TABLE.BloodUnitModification.FromBloodUnitGuid];
				BOL.BloodUnit newUnit = new BOL.BloodUnit(fromGuid);
				this._poolingUnits.Add(newUnit);

				this._originalHasB2Set = (this._originalHasB2Set) ? true : newUnit.HasB2Attribute;

				//Save the original pooling unit guids so we can implement BR_24.28
				this._originalPoolingUnits.Add(fromGuid);
				this._remainingOriginalUnits.Add(fromGuid);
			}

			//We have all the pooling units, so grab the ExpirationDateRule
			//CR2735 - this.GetExpirationDateRuleValues(null); calculated in real time now, not needed

			//Grab the existing Equipment - do this here instead of ModifiedUnit to avoid BR conflicts
			this._dtOriginalPoolEquipment = DAL.BloodUnitModification.GetModificationEquipment(this._pooledUnit.BloodUnitModificationGuid);
			foreach(System.Data.DataRow dr in this._dtOriginalPoolEquipment.Rows)
			{
				BOL.Equipment addEquip = new BOL.Equipment((System.Guid)dr[TABLE.Equipment.EquipmentGuid]);
				this._pooledUnit.EquipmentList.Add(addEquip);
			}

			//Grab the existing supplies
			this._dtOriginalPoolSupply = DAL.BloodUnitModification.GetModificationSupplies(this._pooledUnit.BloodUnitModificationGuid);
			foreach(System.Data.DataRow dr in this._dtOriginalPoolSupply.Rows)
			{
				BOL.Supply addSupply = new BOL.Supply((System.Guid)dr[TABLE.Supply.SupplyGuid]);
				
				//Add the Supplies, SCDWafer, and Container
				if (addSupply.SupplyTypeId == (int)Common.SupplyType.SCDWafer)
				{
					this._pooledUnit.SCDWafer = addSupply;
				}
				else
				{
					if (addSupply.SupplyTypeId == (int)Common.SupplyType.Containers)
					{
						this._pooledUnit.Container = addSupply;
					}
					else
					{
						this._pooledUnit.SupplyList.Add(addSupply);
					}
				}
			}
			this._pooledUnit.ValidInvalidChanged += new EventHandler(this.HandleModifiedUnitStateChange);
		}

		/// <summary>
		/// protected needs XML tags
		/// </summary>
		/// <param name="dtRow"></param>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6985"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>Null</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6986"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// I hate XML tags
		/// </summary>
		/// <param name="dtRow"></param>
		/// <returns></returns>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			return null;
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6987"> 
		///		<ExpectedInput>int</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6988"> 
		///		<ExpectedInput>INvalid int</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Modifies a current configuration of splits
		/// </summary>
		/// <param name="newSplitNumber"></param>
		public void SplitAgain(int newSplitNumber)
		{
			if (newSplitNumber > this.MaximumSplitUnitsAllowed)
			{
				throw new ArgumentException();
			}

			//Just to avoid doing any uneccesary work
			if (newSplitNumber == this.SplitCount) return;

			//If the newSplitNumber is less than what we have, we need to remove  units
			if (newSplitNumber < this.SplitCount)
			{
				while (this.SplitCount > newSplitNumber)
				{
					this.SplitUnits.RemoveAt(this.SplitCount - 1);
				}				
			}
				//Otherwise, we have to add new units
			else
			{
				while (newSplitNumber > this.SplitCount)
				{
					this.AddUnitToSplit(this.SplitCount);
					((BOL.ModifiedUnit)this.SplitUnits[this.SplitCount-1]).SetInitialBrokenRules(this._formName, this._ruleSet);
				}
			}
			this.CalculateSplitProperties();
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6989"> 
		///		<ExpectedInput>index</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6990"> 
		///		<ExpectedInput>Invalid index</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Sets the original container indicator of the unit and unsets it for the 
		/// </summary>
		/// <param name="unitIndex"></param>
		/// <param name="setValue"></param>
		public void SetOriginalContainerIndicator(int unitIndex, bool setValue)
		{
			if (unitIndex > this._splitUnits.Count)
				throw new ArgumentException();

			if (this.ModificationMethod == Common.ModifyUnitMethod.Closed)
			{
				foreach(BOL.ModifiedUnit splitUnit in this._splitUnits)
				{
					splitUnit.OriginalContainerIndicator = setValue;
				}
			}
			else
			{
				if (setValue)
				{
					int ndx=0;
					foreach(BOL.ModifiedUnit splitUnit in this._splitUnits)
					{
						splitUnit.OriginalContainerIndicator = (ndx == unitIndex);
						ndx++;
					}
				}
				else
				{
					((BOL.ModifiedUnit)this._splitUnits[unitIndex]).OriginalContainerIndicator = setValue;
				}
			}

		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6991"> 
		///		<ExpectedInput>Form, Ruleset</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6992"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Sets the control children broken rules so the UI only needs to make one call
		/// </summary>
		/// <param name="formName"></param>
		/// <param name="ruleSet">Useless - but need it to override</param>
		public new void SetInitialBrokenRules(string formName, string ruleSet)
		{
			this._formName = formName;
			string addMethod = "Open";

			if (this.ModificationMethod == Common.ModifyUnitMethod.SterileConnectingDevice)
				addMethod = "SCD";
			if (this.ModificationMethod == Common.ModifyUnitMethod.Closed)
				addMethod = "Closed";

			switch (this.ProductModificationCode)
			{
				case Common.ProductModification.SplitAdult:
			
					//The control takes care of the volume
					base.SetInitialBrokenRules(formName, "SplitControl" + addMethod);

					//The unit's have wafer, container
					foreach(BOL.ModifiedUnit splitUnit in this._splitUnits)
					{						
						this._ruleSet = "SplitModifiedUnit" + addMethod;

						splitUnit.SetInitialBrokenRules(formName, this._ruleSet);
					}

					break;

				case Common.ProductModification.Pooled:
				case Common.ProductModification.PoolThawCryo:
					base.SetInitialBrokenRules(formName, "PoolControl" + addMethod);
					this._ruleSet = "PoolModifiedUnit" + addMethod;
					this.PooledUnit.SetInitialBrokenRules(formName, this._ruleSet);
					break;
				case Common.ProductModification.EditPool:
					base.SetInitialBrokenRules(formName, "EditPoolControl");
					this._ruleSet = "EditPoolModifiedUnit";
					this.PooledUnit.SetInitialBrokenRules(formName, this._ruleSet);
					break;
				default:
					base.SetInitialBrokenRules(formName, "ModifyControl" + addMethod);
					this._ruleSet = "ModifyModifiedUnit" + addMethod;
					this.ModifiedUnit.SetInitialBrokenRules(formName, this._ruleSet);
					break;
			}
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6993"> 
		///		<ExpectedInput>FormName</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6994"> 
		///		<ExpectedInput>Invalid FormName</ExpectedInput>
		///		<ExpectedOutput>EmptyString</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates a giant broken rules string of the control and the modified unit to pass back to the UI
		/// </summary>
		/// <param name="formName">Name of the UI calling form</param>
		/// <returns></returns>
		public new string GetBrokenRulesMessage(string formName)
		{
			System.Text.StringBuilder sb = new System.Text.StringBuilder();
			string childRules;

			//Put the "control" rules in first, then the "children"
			sb.Append(base.GetBrokenRulesMessage(formName));
			
			if (!(sb.Length == 0))
				sb.Append(Environment.NewLine);

			if (this.IsSplit)
			{
				foreach(BOL.ModifiedUnit splitUnit in this._splitUnits)
				{
					childRules = splitUnit.GetBrokenRulesMessage(formName);
					if (childRules.Trim() != string.Empty)
					{
						sb.Append(splitUnit.NewUnit.EyeReadableUnitId);
						sb.Append(Environment.NewLine);
						sb.Append(childRules);
						sb.Append(Environment.NewLine);
					}
				}
			}
			else if (this.IsPool)
			{
				sb.Append(this.PooledUnit.GetBrokenRulesMessage(formName));
			}
			else
			{
					sb.Append(this.ModifiedUnit.GetBrokenRulesMessage(formName));
			}
			return sb.ToString();
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6995"> 
		///		<ExpectedInput>Guid, Int</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6996"> 
		///		<ExpectedInput>Invalid Guid, int</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Splits a blood unit into the number of distinct units as specified by the input parameters
		/// </summary>
		/// <param name="bloodUnitGuid">Bloodunit to be split</param>
		/// <param name="numberOfSplits">Parts to split the unit into</param>
		public void SplitUnit(System.Guid bloodUnitGuid, int numberOfSplits)
		{
			try
			{
				//Save our original - this gets the latest information (in case something may have changed)
				this._splittingUnit = new BOL.BloodUnit(bloodUnitGuid);
				this._originalHasB2Set = this._splittingUnit.HasB2Attribute;

				//Clear out any previous splits
				this._splitUnits.Clear();

				for (int ndx = 0; ndx < numberOfSplits; ndx++)
				{
					this.AddUnitToSplit(ndx);
				}
				this.CalculateSplitProperties();
			}
			catch (Exception err)
			{
				throw new BOL.BusinessObjectException(err.Message);
			}
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6997"> 
		///		<ExpectedInput>BloodUnit</ExpectedInput>
		///		<ExpectedOutput>ArrayList</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6998"> 
		///		<ExpectedInput>Invalid Unit</ExpectedInput>
		///		<ExpectedOutput>ArrayList</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Adds a unit to our collection of pooling units after validating it against business rules
		/// This method assumes that the ValidateBusinessRules method has been run already against
		/// the incoming unit
		/// Implements BR_1.40, BR_12.06, BR_5.02,BR_24.05a,BR_24.05b
		/// </summary>
		/// <param name="addingUnit"></param>
		/// <returns></returns>
		public System.Collections.ArrayList ValidateAddingPoolUnit(BOL.BloodUnit addingUnit)
		{
			bool alreadyExists = false;

			System.Collections.ArrayList actionItems = new System.Collections.ArrayList();

			//Just in case the UI is telling us to add the same item twice...because the worklistbuilder
			//is such a piece of work... we'll just exit gracefully
			foreach(BOL.BloodUnit poolUnit in this._poolingUnits)
			{
				//BR_1.40, BR_12.06, BR_5.02
				if(poolUnit.BloodUnitGuid == addingUnit.BloodUnitGuid)
				{
					alreadyExists = true;
					break;
				}
			}

			//BR_24.05a - Can't add the same unit twice to a pool
			if (alreadyExists)	return (actionItems);

			//New Unit Can't change target BloodProduct during an edit
			if (this.ProductModificationCode == Common.ProductModification.EditPool)
			{
				bool bpPassed = false;
				string bpGuids = this.GetBloodProductGuidArray() + "," + addingUnit.BloodProductGuid.ToString();
				System.Data.DataTable dtTargets = this.GetTargetBloodProducts(bpGuids);
				foreach(System.Data.DataRow dr in dtTargets.Rows)
				{
					if ((System.Guid)dr[TABLE.BloodUnit.BloodProductGuid] == this._pooledUnit.NewUnit.BloodProductGuid)
					{
						bpPassed = true;
						break;
					}
				}

				if (!bpPassed)
				{
					//This is half of BR_24.28, other half is in ValidatePoolRules
					actionItems.Add(new Common.ActionItem(Common.VerifyAction.Cancel, Common.StrRes.ValidMsg.UC024.BR28AddingUnitWillChangeTarget(addingUnit.EyeReadableUnitId, addingUnit.ProductType.ProductTypeText ).ResString));
					return(actionItems);
				}
			}
			
			//BR_24.05b - Can't pool a split unit
			if (BOL.BloodUnitModification.HasUnitBeenSplit(addingUnit))
			{
				actionItems.Add(new Common.ActionItem(Common.VerifyAction.Cancel, Common.StrRes.ValidMsg.UC024.BR05CantPoolASplit(addingUnit.EyeReadableUnitId, addingUnit.ProductType.ProductTypeText ).ResString));
				return(actionItems);
			}

			//If it's our first unit, we don't have any validation to do
			if (this._poolingUnits.Count == 0)
			{
				return(actionItems);
			}
			else
			{
				Common.ActionItem actionResult = this.ValidatePoolRules(addingUnit);

				if (actionResult.ReturnAction != Common.VerifyAction.Continue)
					actionItems.Add(actionResult);
			}

			return(actionItems);
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/26/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2447"> 
		///		<ExpectedInput>Valid GuidArray</ExpectedInput>
		///		<ExpectedOutput>DataTable containing target blood products</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2448"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a data table of target blood products. Implements BR_26.12 
		/// </summary>
		/// <param name="bloodProductGuidArray">Comman delimited arraylist of 'from' bloodproductguids</param>
		/// <returns></returns>
		public DataTable GetTargetBloodProducts(string bloodProductGuidArray)
		{
			char modificationCode = char.MinValue;
			Common.ModifyUnitMethod modificationMethod = this.ModificationMethod;

			modificationCode = Common.Utility.GetProductModificationCharFromEnum(this.ProductModificationCodeBase);

			if (this.ModificationMethod == Common.ModifyUnitMethod.SterileConnectingDevice)
			{
				modificationMethod = this.NewContainerWeldIndicator ? Common.ModifyUnitMethod.Closed : Common.ModifyUnitMethod.Open;
			}

			return DAL.BloodProduct.GetTargetBloodProducts(bloodProductGuidArray, modificationCode, modificationMethod);
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6999"> 
		///		<ExpectedInput>BloodUnit</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7000"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This actually does the adding of the unit to the pool
		/// </summary>
		/// <param name="addingUnit"></param>
		/// <returns></returns>
		public void AddUnitToPool(BOL.BloodUnit addingUnit)
		{

			//Just in case the UI is telling us to add the same item twice...because the worklistbuilder
			//is such a piece of work... we'll just exit gracefully
			foreach(BOL.BloodUnit poolUnit in this._poolingUnits)
			{
				if(poolUnit.BloodUnitGuid == addingUnit.BloodUnitGuid)
				{
					return;
				}
			}

			this._poolingUnits.Add(addingUnit);

			this._originalHasB2Set = (this._originalHasB2Set) ? true : addingUnit.HasB2Attribute;

			if (this._poolingUnits.Count != 1)	//Since we add the unit to the collection above, check for == 1
			{
				//If we're adding an original unit (that was earlier removed)... rebuild our array
				if (this.ProductModificationCode == Common.ProductModification.EditPool)
				{
					if (this.IsOriginalUnit(addingUnit.BloodUnitGuid))
					{
						if (this._remainingOriginalUnits.IndexOf(addingUnit.BloodUnitGuid) < 0)
							this._remainingOriginalUnits.Add(addingUnit.BloodUnitGuid);
					}
				}
			}

		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7001"> 
		///		<ExpectedInput>BloodUnitGuid</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7002"> 
		///		<ExpectedInput>Invalid BloodUnitGuid</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Checks the blood unit guid being addded.  If it doesn't exist in our temp Array, but does
		/// exist in the orignalPoolUnits array, then it needs to get added to temmp Array so we know
		/// how many of the original units are left in the Edit Pool
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		public bool IsOriginalUnit(System.Guid bloodUnitGuid)
		{
			return (this._originalPoolingUnits.IndexOf(bloodUnitGuid) != -1);
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7003"> 
		///		<ExpectedInput>BloodUnitGuid</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7004"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Removes a unit from a pooling.  It won't raise any flags when trying to remove an invalid unit
		/// Implements BR_24.28, 24.27, BR_24.21
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		public void RemoveUnitFromPool(System.Guid bloodUnitGuid)
		{
			//BR_24.28 - we can't remove ALL of the original pooling units for an Edit mod
			if (this.ProductModificationCode == Common.ProductModification.EditPool)
			{
				//If the guid is one of our "original" guids
				if (this._remainingOriginalUnits.IndexOf(bloodUnitGuid) > -1)
				{
					//BR_24.27 -- If it's the last one selected, raise an exception to prevent it from being removed
					if (this._remainingOriginalUnits.Count == 1)
						throw new BOL.BusinessObjectException(Common.StrRes.SysErrMsg.UC024.CannotRemoveAllOriginalUnits().ResString);
					else
					{
						this._remainingOriginalUnits.Remove(bloodUnitGuid);
					}
				}
			}

			//Loop through our array of pooling units and remove the unit(if found) from the collection
			BOL.BloodUnit removeUnit = null;
			foreach(BOL.BloodUnit poolUnit in this._poolingUnits)
			{
				if (poolUnit.BloodUnitGuid == bloodUnitGuid)
				{
					removeUnit = poolUnit;
					break;
				}
			}
            
			if (removeUnit != null)
			{
				this._poolingUnits.Remove(removeUnit);
			}
		}

		/// <summary>
		/// Implements BR_25.11, BR_25.12, BR_25.06, BR_25.07
		/// Adds a unit to the split collection, and sets some of it's default values
		/// </summary>
		/// <param name="index"></param>
		private void AddUnitToSplit(int index)
		{
			//BR_25.11 - the modfiied unit will be the same product type as the parent (because we're creating the new unit with the splitting unit's guid)
			//BR_25.12 - We create the child unit using the original unit's guid so we automatically get
			//alot of the original unit's properties assigned for us
			//CR2002 - Removing Open/Closed Expiration int values, variables are now public access
			BOL.ModifiedUnit child = new BOL.ModifiedUnit(	this._splittingUnit.BloodUnitGuid, this);

			//BR_25.06 -- Codabar units get new EyeReadables
			if (this.SplittingUnit.BarcodeType == Common.BarcodeType.Codabar)
			{
				child.NewUnit.SetEyeReadableUnitId(this._splittingUnit.EyeReadableUnitId + this.GetAlphabeticCharacterByIndex(index), false);
			}
			//BR_25.07 - ISBT units get new UnitDivisionId's
			//CR 2756
			string splittingUnitDivID = this.SplittingUnit.UnitDivisionId;
			if (this.SplittingUnit.BarcodeType == Common.BarcodeType.ISBT128)
			{
				//If the unit divisionid is anything but 00, then it's been split
				if(splittingUnitDivID != "00")
				{
					//When a previously divided unit, where the 7th character is not a numeric, the 8th 
					//character is used to designate additional division of the unit by incrementing 
					//alphabetically in lower case, for example, if a product code ending in A0 is 
					//split into 2 units, the new product codes will end in Aa, Ab, until Az.  
					if(char.IsLetter(splittingUnitDivID.Substring(0,1), 0))//this line is checking the 7th character is not numeric
					{
						//if the split unit that they are splitting has not been split in VBECS
						if(BOL.BloodUnitModification.CheckBUMTableForModificationRecord(this.SplittingUnit)== false)
						{
							child.NewUnit.UnitDivisionId = String.Concat(splittingUnitDivID.Substring(0,1), this.GetLowerAlphabeticCharacterByIndex(index));
						}
						else
						{
							throw new BOL.BusinessObjectException(Common.StrRes.SysErrMsg.UC025.UnitAlreadySplit().ResString);
						}
					}
					else
					{
						throw new BOL.BusinessObjectException( Common.StrRes.InfoMsg.UC025.SplitNotAllowed().ResString);
					}
				}
					//BR_25.07 When a previously undivided unit (00) is divided, character 7 of the 8-digit ISBT 128
					//product code is incremented alphabetically in upper case, resulting in characters 7 
					//and 8 as follows; A0, B0, C0, D0 until Z0.
				else
				{
					child.NewUnit.UnitDivisionId = DetermineDivisionID(index);
				}
			}


			//Add our event to listen to volume changes
			child.UnitVolumeChanged += new EventHandler(this.SplitUnitVolumeChanged);
			child.ValidInvalidChanged += new EventHandler(this.HandleModifiedUnitStateChange);

			this._splitUnits.Add(child);
		}
		/// <summary>
		/// This method calculates the appropraite alpha character to use in a situation where
		/// a split unit was already recieven on the base unit that was modified outside of VBECS
		/// </summary>
		/// <param name="counter"></param>
		/// <returns></returns>
		private string DetermineDivisionID(int counter)
		{
			string unitDivisionID = String.Concat(this.GetAlphabeticCharacterByIndex(counter),"0");
			
			foreach( object obj in this._splitUnits)
			{
				BOL.ModifiedUnit child = (BOL.ModifiedUnit) obj;
				if(child.NewUnit.UnitDivisionId == unitDivisionID)
				{
					return DetermineDivisionID(counter + 1);
				}
			}
			System.Guid unitGuid = new Guid();
			if(BOL.BloodUnit.BloodUnitExists(this._splittingUnit.EyeReadableUnitId, this._splittingUnit.BloodProductGuid, this._splittingUnit.ProductCode + this._splittingUnit.DonationType.DonationTypeCode + unitDivisionID , ref unitGuid))
			{
				return DetermineDivisionID(counter + 1);
			}
			else
			{
				return unitDivisionID;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7010"> 
		///		<ExpectedInput>UpdateFunction</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7011"> 
		///		<ExpectedInput>Inavlid Pooling Units Count</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Pools all the units that have been added to our _poolingUnits collection
		/// This is gunna get very messy
		/// IMplements - BR_24.11, BR_24.12
		/// </summary>
        /// <param name="lockController">LOCKING420 - Used to handle locking</param>
		/// <returns></returns>
		public void PoolUnits(BOL.LockManager lockController)
		{
			//Should never hit here, but JIC
			if (this._poolingUnits.Count < 2)
			{
				throw new BOL.BusinessObjectException("Invalid pooling attempt.  A pool must contain at least two units.");
			}

			//this.GetExpirationDateRuleValues(null);		//CR2735 calculatd in real time, not needed	//Now that I have my pooling units, it's ok to do this

			if (this.ProductModificationCode != Common.ProductModification.EditPool)
			{
				//Get the BloodUniGUID of our first pooling unit to create the pooledunit from
				System.Guid intialUnitGuid = ((BOL.BloodUnit)this._poolingUnits[0]).BloodUnitGuid;

				//Create our new pooled unit from initial to populate a lot of the values we'll need
				//CR2002 - Removing Open/Closed Expiration int values, variables are now public access
				this._pooledUnit = new BOL.ModifiedUnit(intialUnitGuid, this);
			}

			this.CalculatePooledProperties();
			
			if (this.ProductModificationCode != Common.ProductModification.EditPool)
			{
				//Not sure if I should be setting both LabelUnitId and EyeReadableUnit Id
				this._pooledUnit.NewUnit.SetEyeReadableUnitId(this.GetNewPoolId(lockController), false);
				//CR1993 - Set the LabelUnitId as well as EyeReadableUnitId, since LabelUnitId is used 
				//on the label face
				this._pooledUnit.NewUnit.SetLabelUnitId(this._pooledUnit.NewUnit.EyeReadableUnitId, false);
			}

			this._pooledUnit.NewUnit.OriginalVolume = this.TotalVolumeOfPoolingUnits;

			this._pooledUnit.UnitVolumeChanged += new EventHandler(this.PoolUnitVolumeChanged);
			this._pooledUnit.ValidInvalidChanged += new EventHandler(this.HandleModifiedUnitStateChange);
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7012"> 
		///		<ExpectedInput>BloodUnitGuid</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7013"> 
		///		<ExpectedInput>Invalid BloodUnitGuid</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Modify a unit
		/// </summary>
		/// <param name="bloodUnitGuid">GUID of the bloodunit to Modify</param>
		public void ModifyUnit(	System.Guid bloodUnitGuid)
		{
			//Load our original unit
			this._originalUnit = new BOL.BloodUnit(bloodUnitGuid);
			//this.GetExpirationDateRuleValues(null);	//CR2735 - Caulculatd in real time , not needed		//Now that I have original unit, it's ok to do this

			this._originalHasB2Set = this._originalUnit.HasB2Attribute;

			//CR2002 - Removing Open/Closed Expiration int values, variables are now public access
			this._modifiedUnit = new BOL.ModifiedUnit(	bloodUnitGuid, this);

			//Set default volumes for VolumeReduction modifications according to BR_26.41
			if (this.ProductModificationCode == Common.ProductModification.VolumeReduced)
			{
				this._modifiedUnit.NewUnit.OriginalVolume --;
				this._modifiedUnit.NewUnit.DiscardedPlasmaVolume = 1;
			}

			//Adding the ModificationCost to the UnitCost charge
			if (this._modifiedUnit.NewUnit.BloodUnitFinance.UnitCost == decimal.MinValue)
				this._modifiedUnit.NewUnit.BloodUnitFinance.UnitCost = this.ModificationCost;
			else
				this._modifiedUnit.NewUnit.BloodUnitFinance.UnitCost += this.ModificationCost;

			//Moved to modify unit. BR_26.41 change
			this._modifiedUnit.ValidInvalidChanged += new EventHandler(this.HandleModifiedUnitStateChange);
			this._modifiedUnit.UnitVolumeChanged += new EventHandler(this.ModifyUnitVolumeChanged);
			this._modifiedUnit.NewUnit.DiscardedPlasmaVolumeChanged += new EventHandler(this.ModifyUnitVolumeChanged);
		}


		/// <summary>
		/// Calls spGetTargetProducts to grab the expiration date rules -- TT_26.02
		/// Returns an ArrayList 2 wide [0,1] =  0 = Closed, 1 = Open -- redesign for CR2735
		/// CR 2735 - now makes this public - ruh roh! 
		/// </summary>
		public ArrayList GetExpirationDateRuleValues(BOL.BloodProduct targetProduct)
		{
			System.Collections.ArrayList ruleValues = new ArrayList();

			//Set the default
			int openExpirationRule = 0;
			int closedExpirationRule = 0;

			//			//CR2002 - Need to know the BASE modification code here, for Pool Edit dates to be calculated properly
			//This method is only used after we know all our source products, so we can static this table to save roundtrip calls
			if (this._dtExpirationRules == null)
				this._dtExpirationRules = DAL.BloodProduct.GetTargetBloodProducts(this.GetBloodProductGuidArray(), Common.Utility.GetProductModificationCharFromEnum(this.ProductModificationCodeBase), Common.ModifyUnitMethod.None);

			//I could make thsese static, but there's no time! plus I doubt it'd have savings with the extra IF checks needed
			System.Data.DataRow[] drClosedNoTarget = this._dtExpirationRules.Select("MethodSelect = 0");
			System.Data.DataRow[] drOpenNoTarget = this._dtExpirationRules.Select("MethodSelect = 1");

			//Grab the first record as another defualt (in case the product isn't found (or set yet)
			if (drClosedNoTarget.Length > 0)
				closedExpirationRule = (int)drClosedNoTarget[0][Common.VbecsTables.PermittedProductModification.ExpirationDateRule];
			
			if (drOpenNoTarget.Length > 0)
				openExpirationRule = (int)drOpenNoTarget[0][Common.VbecsTables.PermittedProductModification.ExpirationDateRule];

			//If a target is selected, this could alter our results, since target products can have different stroage times
			if (targetProduct != null)
			{
				System.Data.DataRow[] drClosedTarget = this._dtExpirationRules.Select("MethodSelect = 0 AND BloodProductGuid = '" + targetProduct.BloodProductGuid.ToString() + "'");
				System.Data.DataRow[] drOpenTarget = this._dtExpirationRules.Select("MethodSelect = 1 AND BloodProductGuid = '" + targetProduct.BloodProductGuid.ToString() + "'");

				if (drClosedTarget.Length > 0)
					closedExpirationRule = (int)drClosedTarget[0][Common.VbecsTables.PermittedProductModification.ExpirationDateRule];
			
				if (drOpenTarget.Length > 0)
					openExpirationRule = (int)drOpenTarget[0][Common.VbecsTables.PermittedProductModification.ExpirationDateRule];
			}

			ruleValues.Add(closedExpirationRule);
			ruleValues.Add(openExpirationRule);
			return(ruleValues);

		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7014"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7015"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves a list of blood units that are modifiable based on input parameters/divisioncode
		/// The sproc uses BR_26.33 to only select units in Limited, Available, Assigned or XMatched status
		/// BR_25.17 - BloodUnits from any ProductType can be split (this is taken care of in the sproc)
		/// BR_26.43, TT_26.02A
		/// </summary>
		/// <param name="productModificationCode">Modification being performed</param>
		/// <param name="modificationMethod"></param>
		/// <param name="includeAllStatus">When true - will return non LSAC stati</param>
		/// <returns></returns>
		public static DataTable GetModifiableBloodUnits(Common.ProductModification productModificationCode, Common.ModifyUnitMethod modificationMethod, bool includeAllStatus)
		{
			if (productModificationCode == Common.ProductModification.EditPool)
				return DAL.BloodUnitModification.GetPooledBloodUnits();
			else
				return DAL.BloodUnitModification.GetModifiableBloodUnits(productModificationCode, modificationMethod, includeAllStatus);
			
		}


		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/3/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7271"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7272"> 
		///		<ExpectedInput>Invalid Blood Unit Guid</ExpectedInput>
		///		<ExpectedOutput>Empty blood unit modification table</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function returns all the active blood unit modification records with the suppled blood unit as the parent
		/// </summary>
		/// <param name="bloodUnitGuid">Blood Unit Guid</param>
		/// <returns>Blood Unit Modification data table</returns>
		public static DataTable GetBloodUnitModificationsByUnitGuid(Guid bloodUnitGuid)
		{
			return DAL.BloodUnitModification.GetBloodUnitModificationsByUnitGuid(bloodUnitGuid);
		}

		/// <summary>
		/// Static function returns all the active blood unit modification records with the suppled blood unit as the child
		/// </summary>
		/// <param name="bloodUnitGuid">Blood Unit Guid</param>
		/// <returns>Blood Unit Modification data table</returns>
		public static DataTable GetBloodUnitModificationsForChildUnit(Guid bloodUnitGuid)
		{
			return DAL.BloodUnitModification.GetBloodUnitModificationsForChildUnit(bloodUnitGuid);
		}


		/// <summary>
		/// Returns the alphabetic character that corresponds to the index
		/// </summary>
		/// <param name="index"></param>
		/// <returns></returns>
		private string GetAlphabeticCharacterByIndex(int index)
		{
			try
			{
				return "ABCDEFGHIJKLMNOPQRSTUVWXYZ"[index].ToString();
			}
			catch
			{
				throw new BOL.BusinessObjectException(Common.StrRes.InfoMsg.UC025.RanOutOfAlphabet().ResString);
			}
		}
		/// <summary>
		/// Returns the lower case alphabetic character that corresponds to the index
		/// </summary>
		/// <param name="index"></param>
		/// <returns></returns>
		private string GetLowerAlphabeticCharacterByIndex(int index)
		{
			try
			{
			return "abcdefghijklmnopqrstuvwxyz"[index].ToString();
			}
			catch
			{
				throw new BOL.BusinessObjectException(Common.StrRes.InfoMsg.UC025.RanOutOfAlphabet().ResString);
			}
		}
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7393"> 
		///		<ExpectedInput>Split BloodUnit</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7394"> 
		///		<ExpectedInput>Non-Split BloodUnit</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Determines if the asking unit has been split already
		/// </summary>
		/// <param name="bloodUnit"></param>
		/// <returns></returns>
		public static bool HasUnitBeenSplit(BOL.BloodUnit bloodUnit)
		{
			if (bloodUnit.BarcodeType == Common.BarcodeType.Codabar)
			{	
				string oldEyeReadable, lastChar;
				Regex capitalLetterTest = new Regex(@"^[A-Z]{1}");

				oldEyeReadable = bloodUnit.EyeReadableUnitId;
				lastChar = oldEyeReadable.Substring(oldEyeReadable.Length - 1);

				//If there's a capital letter there, it's been split, return false
				if(capitalLetterTest.IsMatch(lastChar))
				{
					return BloodUnitModification.CheckBUMTableForModificationRecord(bloodUnit);
				}
			}
			else
			{
				
				//if the last character of the product code is not zero do not allow split
				if(bloodUnit.UnitDivisionId.Substring(1,1) != "0")
				{
					return true;
				}
				//If the unit divisionid is anything but 00, then it's been split
				if(bloodUnit.UnitDivisionId != "00")
				{
					//returns to not allow split if the unit was modified within VBECS
					return BloodUnitModification.CheckBUMTableForModificationRecord(bloodUnit);
				}
				
			}
			return false;
		}
		//BR_25.21 says that you can now split a split that was shipped in split from another facility 
		//this method allows us to know if a split was split by us.
		private static bool CheckBUMTableForModificationRecord(BOL.BloodUnit bloodUnit)
		{
/*CR3071*/			System.Data.DataTable dtModifications = BOL.BloodUnitModification.GetBloodUnitModificationsForChildUnit(bloodUnit.BloodUnitGuid);
//CR3071			System.Data.DataTable dtModifications = BloodUnitModification.GetBloodUnitModificationsByUnitGuid(bloodUnit.BloodUnitGuid);
			if(dtModifications.Rows.Count > 0)
			{
                foreach(System.Data.DataRow dtRow in dtModifications.Rows)
				{
					char retVal = dtRow[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToCharArray(0,1)[0];
/*CR3071*/			if (retVal == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.SplitAdult))
/*CR3071*/				return(true);
				}
				return false;
			}
			else
			{
				return false;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7279"> 
		///		<ExpectedInput>BloodUnit, DataRow, boolean</ExpectedInput>
		///		<ExpectedOutput>DataRow populated with values</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7280"> 
		///		<ExpectedInput>Invalid DataRow</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///		///<Interfaces></Interfaces>
		/// <summary>
		/// Populates the datarow with new parent information to be written to bloodunit table
		/// Implements BR_24.24
		/// </summary>
		/// <param name="parentUnit">Unit being modified - (parent)</param>
		/// <param name="dtRow">Empty datarow to populate</param>
		/// <returns></returns>
		public System.Data.DataRow LoadParentBloodUnitInformation(BOL.BloodUnit parentUnit, System.Data.DataRow dtRow)
		{
			dtRow[TABLE.BloodUnit.BloodUnitGuid] = parentUnit.BloodUnitGuid;
			dtRow[TABLE.BloodUnit.OriginalVolume] = parentUnit.OriginalVolume;

			if (parentUnit.DiscardedPlasmaVolume > decimal.Zero)
				dtRow[TABLE.BloodUnit.DiscardedPlasmaVolume] = parentUnit.DiscardedPlasmaVolume;

			dtRow[TABLE.BloodUnit.DonationTypeId] = parentUnit.DonationType.DonationTypeId;
			dtRow[TABLE.BloodUnit.RowVersion] = parentUnit.RowVersion;

			return dtRow;
		}
		
		/// <summary>
		/// Copies the BloodUnitAntigen information over from DR to DR
		/// </summary>
		/// <param name="antigen"></param>
		/// <param name="dtRow"></param>
		/// <returns></returns>
		private System.Data.DataRow LoadBloodUnitAntigenInformation(BOL.BloodUnitAntigen antigen, System.Data.DataRow dtRow)
		{

			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitAntigen.AntigenTypeId))
			{
				dtRow[TABLE.BloodUnitAntigen.AntigenTypeId] = antigen.AntigenTypeId;
			}

			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitAntigen.BloodUnitGuid))
			{
				dtRow[TABLE.BloodUnitAntigen.BloodUnitGuid] = antigen.BloodUnitGuid;
			}

			//Field never used - 04/14/2006 - DBA - recommend removal from table
//			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitAntigen.Comments))
//			{
//				dtRow[TABLE.BloodUnitAntigen.Comments] = antigen.Comments;
//			}

			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitAntigen.DivisionCode))
			{
				dtRow[TABLE.BloodUnitAntigen.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			}

			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitAntigen.Positive))
			{
				dtRow[TABLE.BloodUnitAntigen.Positive] = antigen.Positive;
			}
			
			//We're only inserting active antigens
			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitAntigen.RecordStatusCode))
			{
				dtRow[TABLE.BloodUnitAntigen.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
			}

			return(dtRow);
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/12/2004</CreationDate>
		/// <summary>
		/// Populates the incoming datarow populated with all relevant Shipment information
		/// Implements BR_26.39
		/// </summary>
		/// <param name="dtRow"></param>
		/// <param name="shipmentGuid"></param>
		/// <param name="localSupplierGuid"></param>
		/// <param name="targetBloodProductGuid"></param>
		/// <returns>Datarow with shipment information</returns>
		private System.Data.DataRow LoadShipmentInformation(System.Data.DataRow dtRow, System.Guid shipmentGuid, System.Guid localSupplierGuid, System.Guid targetBloodProductGuid)
		{
			if (dtRow.Table.Columns.Contains(TABLE.Shipment.DivisionCode))
				dtRow[TABLE.Shipment.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;

			if(dtRow.Table.Columns.Contains(TABLE.Shipment.ProductShipperGuid))
			{
				BOL.ProductShipperCollection tmpPSC = new BOL.ProductShipperCollection(targetBloodProductGuid);
				BOL.ProductShipper targetShipper = tmpPSC.GetElementByLocalSupplierGuid(localSupplierGuid);

				System.Guid productShipperGuid = System.Guid.NewGuid();
				
				//Since Split's automatically assign the target product type, if it's not active, we do it BTS for them
				if (this.IsSplit)
				{
					if (targetShipper == null)
					{
						BOL.ProductShipper tmpPS = new BOL.ProductShipper(productShipperGuid);
						tmpPS.BloodProductGuid = this.SplittingUnit.BloodProductGuid;
						tmpPS.LocalSupplierGuid = localSupplierGuid;
						tmpPS.ReturnCreditPercentage = 100;
						tmpPS.ProductCost = 0;
						tmpPS.RecordStatusCode = Common.RecordStatusCode.Active;
						
						if (!tmpPS.Insert(Common.UpdateFunction.UC025FrmSplitUnits))
							throw new BOL.BusinessObjectException("Target product could not be activated for your division.");
					}
					else
					{
						if (targetShipper.RecordStatusCode != Common.RecordStatusCode.Active)
						{
							//The record was found, so we need to activate the shipper
							targetShipper.RecordStatusCode = Common.RecordStatusCode.Active;
							if (!targetShipper.Update(Common.UpdateFunction.UC025FrmSplitUnits))
							{
								throw new BOL.BusinessObjectException("Target product could not be activated for your division.");
							}
						}
						productShipperGuid = targetShipper.ProductShipperGuid;
					}
				}
				else
				{
					//Non splitting logic is here...which, if the UI worked properly, SHOULD have a PS 
					if (targetShipper == null)
					{
						//Just in case (to cover the UI's problem)
						throw new BOL.BusinessObjectException("Target product shipper not found.");
					}
					productShipperGuid = targetShipper.ProductShipperGuid;
				}

				//Assign value to the DataColumn
				dtRow[TABLE.Shipment.ProductShipperGuid] = productShipperGuid;
			}

		
			//Code similar to Shipment.GenerateNewInvoiceNumber()
			if(dtRow.Table.Columns.Contains(TABLE.Shipment.InvoiceNumber))
				dtRow[TABLE.Shipment.InvoiceNumber] = this.InvoiceNumber;

			if(dtRow.Table.Columns.Contains(TABLE.Shipment.LocalSupplierGuid))
				dtRow[TABLE.Shipment.LocalSupplierGuid] = localSupplierGuid;

			//BR_26.39 - shipment date is time of the modification
			if(dtRow.Table.Columns.Contains(TABLE.Shipment.ShipmentDate))
				dtRow[TABLE.Shipment.ShipmentDate] = this.ModificationDateTime;

			if(dtRow.Table.Columns.Contains(TABLE.Shipment.ShipmentDirection))
				dtRow[TABLE.Shipment.ShipmentDirection] = Common.Utility.GetShipmentDirectionFromEnum(Common.ShipmentDirection.IncomingShipment);

			if(dtRow.Table.Columns.Contains(TABLE.Shipment.ShipmentGuid))
				dtRow[TABLE.Shipment.ShipmentGuid] = shipmentGuid;

			if(dtRow.Table.Columns.Contains(TABLE.Shipment.RecordStatusCode))
				dtRow[TABLE.Shipment.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Modified);

	
			return dtRow;
		}


		/// <summary>
		/// LoadTargetBloodUnitInformation
		/// </summary>
		/// <param name="modifiedUnit">Unit to strip data from</param>
		/// <param name="newRow"></param>
		/// <returns></returns>
		private System.Data.DataRow LoadTargetBloodUnitInformation(BOL.ModifiedUnit modifiedUnit, System.Data.DataRow newRow)
		{
			//For code reuse - check whether we're populating BloodUnit or Media data
			if(newRow.Table.Columns.Contains(TABLE.BloodUnit.BloodUnitGuid))
			{
				newRow[TABLE.BloodUnit.BloodUnitGuid] = modifiedUnit.NewUnit.BloodUnitGuid;

				if (this.ProductModificationCode == Common.ProductModification.EditPool)
					newRow[TABLE.BloodUnit.RowVersion] = modifiedUnit.NewUnit.RowVersion;
				else
					newRow[TABLE.BloodUnit.RowVersion] = this.OriginalUnit.RowVersion;
			}

			if(newRow.Table.Columns.Contains(TABLE.BloodUnitMedia.BloodUnitMediaGuid))
			{
				newRow[TABLE.BloodUnitMedia.BloodUnitMediaGuid] = modifiedUnit.NewUnit.BloodUnitMedia.BloodUnitMediaGuid;
					
				if (this.ProductModificationCode == Common.ProductModification.EditPool)
					newRow[TABLE.BloodUnitMedia.RowVersion] = modifiedUnit.NewUnit.BloodUnitMedia.RowVersion;
				else
					newRow[TABLE.BloodUnitMedia.RowVersion] = this.OriginalUnit.BloodUnitMedia.RowVersion;
			}

			if(newRow.Table.Columns.Contains(TABLE.BloodUnitMedia.TestingStatusId))
			{
				if (modifiedUnit.NewUnit.BloodUnitMedia.TestingStatusId > 0)
					newRow[TABLE.BloodUnitMedia.TestingStatusId] = modifiedUnit.NewUnit.BloodUnitMedia.TestingStatusId;
			}

			if(newRow.Table.Columns.Contains(TABLE.BloodUnitMedia.DirectedPatientGuid))
			{
				if (modifiedUnit.NewUnit.BloodUnitMedia.DirectedPatientGuid != System.Guid.Empty)
					newRow[TABLE.BloodUnitMedia.DirectedPatientGuid] = modifiedUnit.NewUnit.BloodUnitMedia.DirectedPatientGuid;
			}
				
			if(newRow.Table.Columns.Contains(TABLE.BloodUnitMedia.Comments))
			{
				if (modifiedUnit.NewUnit.BloodUnitMedia.Comments != string.Empty)
				{
					newRow[TABLE.BloodUnitMedia.Comments] = modifiedUnit.NewUnit.BloodUnitMedia.Comments;
				}
			}

			if(newRow.Table.Columns.Contains(TABLE.BloodUnit.DonationTypeId))
				newRow[TABLE.BloodUnit.DonationTypeId] = modifiedUnit.NewUnit.DonationType.DonationTypeId;

			if(newRow.Table.Columns.Contains(TABLE.BloodUnit.BloodProductGuid))
				newRow[TABLE.BloodUnit.BloodProductGuid] = modifiedUnit.TargetBloodProduct.BloodProductGuid;

			if(newRow.Table.Columns.Contains(TABLE.BloodUnit.UnitProductCode))
				newRow[TABLE.BloodUnit.UnitProductCode] = new BOL.BloodProduct(modifiedUnit.NewUnit.BloodProductGuid).ProductCode;

			if(newRow.Table.Columns.Contains(TABLE.BloodUnit.OriginalVolume))
				newRow[TABLE.BloodUnit.OriginalVolume] = modifiedUnit.NewUnit.OriginalVolume;

			if(newRow.Table.Columns.Contains(TABLE.BloodUnitMedia.BloodTypeCode))
				newRow[TABLE.BloodUnitMedia.BloodTypeCode] = modifiedUnit.NewUnit.BloodUnitMedia.AboRh.Abo.ToString();

			if(newRow.Table.Columns.Contains(TABLE.BloodUnitMedia.RhFactorCode))
				newRow[TABLE.BloodUnitMedia.RhFactorCode] = modifiedUnit.NewUnit.BloodUnitMedia.AboRh.RHFactorCode;

			if(newRow.Table.Columns.Contains(TABLE.BloodUnit.DiscardedPlasmaVolume))
			{
				if (modifiedUnit.NewUnit.DiscardedPlasmaVolume > decimal.Zero)
				{
					newRow[TABLE.BloodUnit.DiscardedPlasmaVolume] = modifiedUnit.NewUnit.DiscardedPlasmaVolume;
				}
			}

			if(newRow.Table.Columns.Contains(TABLE.BloodUnitMedia.UnitExpirationDate))
				newRow[TABLE.BloodUnitMedia.UnitExpirationDate] = modifiedUnit.UnitExpirationDate;


			if(newRow.Table.Columns.Contains(TABLE.BloodUnit.RecordStatusCode))
				newRow[TABLE.BloodUnit.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);

			if(newRow.Table.Columns.Contains(TABLE.BloodUnit.ProcessingTechId))
				newRow[TABLE.BloodUnit.ProcessingTechId] = this.ModificationTechId;

			if(newRow.Table.Columns.Contains(TABLE.BloodUnit.DivisionCode))
				newRow[TABLE.BloodUnit.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;

			return(newRow);
		}

		/// <summary>
		/// Invoice Number generated for the spInsertShiment call in Save
		/// </summary>
		private string InvoiceNumber
		{
			get
			{
				return("@" + System.DateTime.Now.ToString("yyMMddhhmmss"));
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7018"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7019"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Saves the modification ordeal 
		/// </summary>
		/// <param name="lastUpdateFunctionId"></param>
		/// <returns></returns>
		public bool Save(Common.UpdateFunction lastUpdateFunctionId)
		{
			bool retVal = false;

			//Save this off for the preparation methods
			this._lastUpdateFunctionId = lastUpdateFunctionId;

			switch (this._productModificationCode)
			{
				case Common.ProductModification.Pooled:
				case Common.ProductModification.PoolThawCryo:
					retVal = (this.PrepareAndSavePool() == 0);
					break;
				case Common.ProductModification.EditPool:
					retVal = (this.PrepareAndSaveEditedPool() == 0);
					break;
				case Common.ProductModification.SplitAdult:
					retVal = (this.PrepareAndSaveSplit() == 0);
					break;
				case Common.ProductModification.Unknown:
					throw new BOL.BusinessObjectException(Common.StrRes.SysErrMsg.Common.GeneralError("ProductModificationCode").ResString);
				default:
					retVal = (this.PrepareAndSaveModification() == 0);
					break;
			}

			return(retVal);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/2/2007</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8462"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Workload Code</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8463"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets the right workload code based on the ProductModificationCodeBase. Made public for CR1441
		/// </summary>
		/// <returns></returns>
		public Common.WorkloadProcessID WorkloadCode
		{
			get
			{
				Common.WorkloadProcessID workloadCode;

				switch (this.ProductModificationCodeBase)
				{
					case Common.ProductModification.Deglycerolized:
						workloadCode = Common.WorkloadProcessID.DeglycerolizeUnit;
						break;
					case Common.ProductModification.Washed:
						workloadCode = Common.WorkloadProcessID.WashUnit;
						break;
					case Common.ProductModification.VolumeReduced:
						workloadCode = Common.WorkloadProcessID.VolumeReduce;
						break;
					case Common.ProductModification.Frozen:
						workloadCode = Common.WorkloadProcessID.FreezeUnit;
						break;
					case Common.ProductModification.Irradiated:
						workloadCode = Common.WorkloadProcessID.IrradiateUnit;
						break;
					case Common.ProductModification.LeukoReduced:
						workloadCode = Common.WorkloadProcessID.LeukoreduceUnit;
						break;
					case Common.ProductModification.Pooled:
					case Common.ProductModification.EditPool:
						workloadCode = Common.WorkloadProcessID.PoolUnit;
						break;
					case Common.ProductModification.Rejuvenated:
						workloadCode = Common.WorkloadProcessID.RejuvenateUnit;
						break;
					case Common.ProductModification.SplitAdult: 
						workloadCode = Common.WorkloadProcessID.SplitUnit;
						break;
					case Common.ProductModification.PoolThawCryo:
						workloadCode = Common.WorkloadProcessID.ThawPoolCryo;
						break;
					case Common.ProductModification.Thawed:
						workloadCode = Common.WorkloadProcessID.Thaw;
						break;
					default:
						workloadCode = Common.WorkloadProcessID.Unknown;
						break;
				}
				return(workloadCode);
			}
		}

		/// <summary>
		/// Collects and Saves the Split Information
		/// </summary>
		/// <returns></returns>
		private int PrepareAndSaveSplit()
		{
			int nCount = 0;
			System.Collections.ArrayList dtArrayList = new System.Collections.ArrayList();
			System.Collections.ArrayList sprocArrayList = new System.Collections.ArrayList();
			
			System.Data.DataTable dtShipment = DAL.BloodUnitModification.GetEmptyShipmentTable();
			System.Data.DataTable dtBloodUnit = DAL.BloodUnit.GetEmptyBloodUnitTableSchema(false);
			System.Data.DataTable dtBloodUnitFinance = DAL.BloodUnitFinance.GetEmptyBloodUnitFinanceTableSchema(false);
			System.Data.DataTable dtBloodUnitMedia = DAL.BloodUnitMedia.GetEmptyBloodUnitMediaTableSchema(false);
			System.Data.DataTable dtBloodUnitStatus = DAL.BloodUnitStatus.GetEmptyBloodUnitStatusTableSchema(false);

			System.Guid []  bloodUnitGuids = new System.Guid [this._splitUnits.Count];		//Needed for workload
			System.Data.DataTable dtBloodUnitShipment = DAL.BloodUnitModification.GetEmptyBloodUnitShipmentTable();
			System.Data.DataTable dtBloodUnitModification = DAL.BloodUnitModification.GetEmptyBloodUnitModificationTable(false);
			System.Data.DataTable dtOrderedUnit = DAL.BloodUnitModification.GetEmptyOrderedUnitTable();

			System.Guid shipmentGuid = System.Guid.NewGuid();
			System.Guid localSupplierGuid = System.Guid.Empty;
			System.Guid targetBPGuid = System.Guid.Empty;
			nCount = this._splitUnits.Count;
			for(int ndx = 0; ndx < nCount; ndx++)
			{
				BOL.ModifiedUnit tmpUnit = (BOL.ModifiedUnit)this._splitUnits[ndx];

				localSupplierGuid = tmpUnit.NewUnit.IncomingShipment.LocalSupplierGuid;
				targetBPGuid = tmpUnit.TargetBloodProduct.BloodProductGuid;

				//*	spInsertMaintainableBloodUnit -- inserts the new "modified" blood units
				dtBloodUnit.Rows.Add(tmpUnit.LoadBloodUnitInformation(dtBloodUnit.NewRow()));
				dtBloodUnitFinance.Rows.Add(tmpUnit.LoadBloodUnitInformation(dtBloodUnitFinance.NewRow()));
				dtBloodUnitMedia.Rows.Add(tmpUnit.LoadBloodUnitInformation(dtBloodUnitMedia.NewRow()));
				dtBloodUnitStatus.Rows.Add(tmpUnit.LoadBloodUnitInformation(dtBloodUnitStatus.NewRow()));

				bloodUnitGuids[ndx] = tmpUnit.ToBloodUnitGuid;

				//* spInsertBloodUnitShipment
				dtBloodUnitShipment.Rows.Add(tmpUnit.LoadBloodUnitShipmentInformation(dtBloodUnitShipment.NewRow(), shipmentGuid));

				//*	spInsertBloodUnitModification -- creates the link between old and "modified" unit
				System.Data.DataRow drModifiedUnit = this.LoadBloodUnitModificationInformation(tmpUnit,dtBloodUnitModification.NewRow(), this.SplittingUnit);
				dtBloodUnitModification.Rows.Add(drModifiedUnit);

				//BR_26.29 - Dont' save patient assignments if the verification failed
				if (!tmpUnit.LabelVerificationFailed)
				{
					//spInsertOrderedUnit - creates patient assignments for each new unit
					if (this._inheritedPatientAssignment[0] != System.Guid.Empty)
					{
						BOL.OrderedUnit tmpOU = new BOL.OrderedUnit(this._inheritedPatientAssignment[0], this._inheritedPatientAssignment[1]);
						tmpOU.BloodUnitGuid = tmpUnit.ToBloodUnitGuid;	//Assign the new unit to the patient
						tmpOU.OrderedUnitGuid = System.Guid.NewGuid();	//Sproc accepts this?! ACK!					
						if (tmpOU.CurrentCrossmatchDate == System.DateTime.MinValue)		//LoadDataRowFromThis defaults to MINVALUE if NULL
						{
							if (dtOrderedUnit.Columns.Contains(TABLE.OrderedUnit.CrossmatchDate))
								dtOrderedUnit.Columns.Remove(TABLE.OrderedUnit.CrossmatchDate);
							if (dtOrderedUnit.Columns.Contains(TABLE.OrderedUnit.CrossmatchResultCode))
								dtOrderedUnit.Columns.Remove(TABLE.OrderedUnit.CrossmatchResultCode);
							if (dtOrderedUnit.Columns.Contains(TABLE.OrderedUnit.CrossmatchTechId))
								dtOrderedUnit.Columns.Remove(TABLE.OrderedUnit.CrossmatchTechId);
						}
						dtOrderedUnit.Rows.Add(tmpOU.LoadDataRowFromThis(dtOrderedUnit.NewRow()));
					}
				}
					
			}

			
			//Regardless of what's happened above, we want to release all of our split unit patient assignments.
			//The splitting unit, should either have it's assignment create in dtOrderedUnit, or no assignment
			System.Data.DataTable dtUpdateOU = BOL.OrderedUnit.GetEmptyOrderedUnitDataTable();

			System.Data.DataTable dtAssignments = BOL.OrderedUnit.GetOrderedUnit(this.SplittingUnit.BloodUnitGuid, Common.LogonUser.LogonUserDivisionCode );
			foreach(System.Data.DataRow dr in dtAssignments.Rows)
			{
				System.Data.DataRow drRelease = dtUpdateOU.NewRow();
				drRelease[TABLE.OrderedUnit.BloodUnitGuid] = this.SplittingUnit.BloodUnitGuid;
				drRelease[TABLE.OrderedUnit.ReleaseTechId] = this.ModificationTechId;
				drRelease[TABLE.OrderedUnit.ReleaseDate] = this.ModificationDateTime;
				drRelease[TABLE.OrderedUnit.LastUpdateUser] = Common.LogonUser.LogonUserName;
				drRelease[TABLE.OrderedUnit.RowVersion] = dr[TABLE.OrderedUnit.RowVersion];
				drRelease[TABLE.OrderedUnit.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
				dtUpdateOU.Rows.Add(drRelease);
			}

			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtUpdateOU, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.ReleaseOrderedUnitsByBloodUnit.StoredProcName);

			//spInsertShipment -- now (as of 2/28/2005) ONE per Modification
			dtShipment.Rows.Add(this.LoadShipmentInformation(dtShipment.NewRow(), shipmentGuid , localSupplierGuid, targetBPGuid));
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtShipment,this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertShipment.StoredProcName);

			//spInsertBloodUnit, spInsertBloodUnitFinance, spInsertBloodUnitMedia, spInsertBloodUnitStatus*********************
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtBloodUnit, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertBloodUnit.StoredProcName);
			
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtBloodUnitFinance, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertBloodUnitFinance.StoredProcName);
			
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtBloodUnitMedia, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertBloodUnitMedia.StoredProcName);

			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtBloodUnitStatus, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertBloodUnitStatus.StoredProcName);

			//Creates OrderedUnit records for all the split units for the modification
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtOrderedUnit, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertOrderedUnit.StoredProcName);

			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtBloodUnitShipment, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertBloodUnitShipment.StoredProcName);

			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtBloodUnitModification, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertBloodUnitModification.StoredProcName);


			System.Data.DataTable dtOriginalTests = BOL.BloodUnitTest.GetBloodUnitTests(this._splittingUnit.BloodUnitGuid, this._splittingUnit.DivisionCode);
			System.Data.DataSet dsOriginalUnitAntigens = DAL.DiscardOrQuarantineBloodUnit.GetAntigenTypingForConsistencyCheck(this._splittingUnit.BloodUnitGuid);
			int guidNdx=0;
			foreach(BOL.ModifiedUnit newUnit in this._splitUnits)
			{
				System.Collections.ArrayList bumGuids = new System.Collections.ArrayList();	//Needed for consistancy with POOL
				bumGuids.Add(newUnit.BloodUnitModificationGuid);

				//*	spInsertModificationSupply  -- inserts any supplies, wafer, containers used in the modification
				System.Data.DataTable dtModificationSupply = newUnit.LoadModificationSupplyInformation(bumGuids, false);				
				dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtModificationSupply, this._lastUpdateFunctionId));
				sprocArrayList.Add(SPROC.InsertModificationSupply.StoredProcName);

				//*	spInsertModificationEquipment  -- inserts any equpiment used in the modification
				System.Data.DataTable dtModificationEquipment = newUnit.LoadModificationEquipmentInformation(bumGuids,false);				
				dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtModificationEquipment, this._lastUpdateFunctionId));
				sprocArrayList.Add(SPROC.InsertModificationEquipment.StoredProcName);
			
				//* spInsertBloodUnitTest******************************************
				System.Data.DataTable dtBloodUnitTest = this.LoadBloodUnitTestInformation(dtOriginalTests, bloodUnitGuids[guidNdx]);
				dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtBloodUnitTest, this._lastUpdateFunctionId));
				sprocArrayList.Add(SPROC.InsertBloodUnitTest.StoredProcName);
			
				//* spInsertBloodUnitAntigens**************************************
				System.Data.DataTable dtBloodUnitAntigens = this.LoadAntigenInformation(dsOriginalUnitAntigens, (BOL.ModifiedUnit)this._splitUnits[guidNdx]);
				dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtBloodUnitAntigens, this._lastUpdateFunctionId));
				sprocArrayList.Add(SPROC.InsertBloodUnitAntigen.StoredProcName);

				//Any exceptions generated by supplies  -- spInsertExceptionModification****************
				dtArrayList.Add(this.GetSupplyExceptionsDataTable(newUnit.Container));
				sprocArrayList.Add(SPROC.InsertExceptionUnitModification.StoredProcName);

				foreach(BOL.Supply tmpSupply in newUnit.SupplyList)
				{
					dtArrayList.Add(this.GetSupplyExceptionsDataTable(tmpSupply));
					sprocArrayList.Add(SPROC.InsertExceptionUnitModification.StoredProcName);
				}

				dtArrayList.Add(this.GetSupplyExceptionsDataTable(newUnit.SCDWafer));
				sprocArrayList.Add(SPROC.InsertExceptionUnitModification.StoredProcName);
				guidNdx++;
			}

			//*spInsertExceptionModification********************************************
			dtArrayList.Add(this.GetExceptionsDataTable(this.Exceptions));
			sprocArrayList.Add(SPROC.InsertExceptionUnitModification.StoredProcName);

			dtArrayList.Add(this.GetExceptionsDataTable(this.LabelExceptions));
			sprocArrayList.Add(SPROC.InsertExceptionLabelMismatch.StoredProcName);

			//* spInsertWorkloadEvents*****************************************
			//CR1982 - Workload should only be applied to the splitting unit
			System.Guid[] workloadGuid = {this.SplittingUnit.BloodUnitGuid};
			System.Data.DataTable dtWorkload = BOL.WorkloadEvent.GenerateWorkloadData(this.WorkloadCode, workloadGuid, Common.WorkloadTransactionType.Unit, false);
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtWorkload, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertWorkloadEvents.StoredProcName);

			//getting all patients assigned to a unit
			DataTable dtPatients = DAL.PatientAssignments.GetPatientsAssignmentDetails(this._splittingUnit.BloodUnitGuid);

			DataTable dtBCEMessages = null;
			DataTable dtTempBCEMessages = null;

			//CR 2941
			//CR 2945
			//generate table with BCE_MESSAGE
			//The Unit is split, send BCE message releasing all unit assignments

			Guid assignedPatientGuid = Guid.Empty;

			for (int j=0; j< dtPatients.Rows.Count; j++)
			{
				//CR 3015
				assignedPatientGuid = (Guid)dtPatients.Rows[j][Common.VbecsTables.Patient.PatientGuid];
				if (dtBCEMessages == null)
				{
					dtBCEMessages = OrderedUnit.UpdateBceForRelease(this._splittingUnit.BloodUnitGuid, assignedPatientGuid, false);
				}
				else
				{
					dtTempBCEMessages = OrderedUnit.UpdateBceForRelease(this._splittingUnit.BloodUnitGuid, assignedPatientGuid, false);
					if (dtTempBCEMessages.Rows.Count > 0)
						dtBCEMessages.Rows.Add(dtTempBCEMessages.Rows[0].ItemArray);
				}
			}

			if (dtBCEMessages != null)
			{
				if (dtBCEMessages.Rows.Count > 0)
				{
					dtArrayList.Add(dtBCEMessages);
					sprocArrayList.Add(SPROC.HL7InsertMessageLog.StoredProcName);
				}
			}
			//BCE_MESSAGE end
			
			int result = (new Common.StoredProcedure().TransactionalGetValue(sprocArrayList, dtArrayList));

			//send BCE message reserving a new unit only if the user decided to move unit assignment during modification and 
			//if the label verification did not fail
			
			if (result == 0)
			{
				nCount = this._splitUnits.Count;
				for(int i = 0; i < nCount; i++)
				{
					BOL.ModifiedUnit tmpUnit = (BOL.ModifiedUnit)this._splitUnits[i];
					if (tmpUnit.LabelVerificationFailed == false)
					{
						if (this._inheritedPatientAssignment[0] != System.Guid.Empty)
						{
							System.Guid patientGuid = this._inheritedPatientAssignment[1];
							//CR 3015
							Guid orderedComponentGuid = Guid.Empty;
							DataRow[] drAssignments = dtAssignments.Select(Common.VbecsTables.Patient.PatientGuid + " = '" + patientGuid.ToString() + "'");
							if (drAssignments.Length > 0)
								orderedComponentGuid = (Guid)drAssignments[0][Common.VbecsTables.OrderedUnit.OrderedComponentGuid];
							OrderedUnit.UpdateAndSaveBce(tmpUnit.ToBloodUnitGuid, patientGuid, orderedComponentGuid, "", Common.HL7BPDispenseStatus.RS);
						}
					}
				}
			}
			return result;
		}

		/// <summary>
		/// Editing Pool
		/// Implements BR_24.24, BR_24.06, BR_26.29
		/// </summary>
		/// <returns></returns>
		private int PrepareAndSaveEditedPool()
		{
			System.Collections.ArrayList bumGuids = new System.Collections.ArrayList();
			System.Collections.ArrayList dtArrayList = new System.Collections.ArrayList();
			System.Collections.ArrayList sprocArrayList = new System.Collections.ArrayList();			

			//BR_26.29 - Dont save assignment is the verification failed
			if (!this.PooledUnit.LabelVerificationFailed)
			{
				//spInsertOrderedUnit - creates patient assignment for the new unit - using the existing info as a template
				System.Data.DataTable dtOrderedUnit = DAL.BloodUnitModification.GetEmptyOrderedUnitTable();

				if (this._inheritedPatientAssignment[0] != System.Guid.Empty)
				{
                    //CR2650 - check to see if the Pool is already assigned to the patient
                    bool createAssignment = true;
                    DataTable poolAssignments = DAL.PatientAssignments.GetPatientsAssignmentDetails(this.PooledUnit.ToBloodUnitGuid);

                    //Check to see if the patient is already assigned to the pool - if it is, then DONT create a new one
                    foreach (DataRow drAssignment in poolAssignments.Rows)
                    {
                        if (drAssignment[TABLE.Patient.PatientGuid].ToString() == this._inheritedPatientAssignment[1].ToString())
                            createAssignment = false;
                    }

                    if (createAssignment)
                    {
                        BOL.OrderedUnit tmpOU = new BOL.OrderedUnit(this._inheritedPatientAssignment[0], this._inheritedPatientAssignment[1]);
                        tmpOU.BloodUnitGuid = this.PooledUnit.ToBloodUnitGuid; //Assign the new unit to the patient
                        tmpOU.OrderedUnitGuid = System.Guid.NewGuid(); //Sproc accepts this?! ACK!
                        if (tmpOU.CurrentCrossmatchDate == System.DateTime.MinValue) //LoadDataRowFromThis defaults to MINVALUE if NULL
                        {
                            if (dtOrderedUnit.Columns.Contains(TABLE.OrderedUnit.CrossmatchDate))
                                dtOrderedUnit.Columns.Remove(TABLE.OrderedUnit.CrossmatchDate);
                            if (dtOrderedUnit.Columns.Contains(TABLE.OrderedUnit.CrossmatchResultCode))
                                dtOrderedUnit.Columns.Remove(TABLE.OrderedUnit.CrossmatchResultCode);
                            if (dtOrderedUnit.Columns.Contains(TABLE.OrderedUnit.CrossmatchTechId))
                                dtOrderedUnit.Columns.Remove(TABLE.OrderedUnit.CrossmatchTechId);
                        }
                        dtOrderedUnit.Rows.Add(tmpOU.LoadDataRowFromThis(dtOrderedUnit.NewRow()));

                        dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtOrderedUnit, this._lastUpdateFunctionId));
                        sprocArrayList.Add(SPROC.InsertOrderedUnit.StoredProcName);
                    }
				}
			}
			else		//Release the Units if the label unit modification failed
			{
				System.Collections.ArrayList unitArray = new ArrayList();
				unitArray.Add(this.PooledUnit.NewUnit);
				System.Data.DataTable dtOrderedUnit = BOL.OrderedUnit.GetOrderedUnitDataTableForReleaseOfPatientAssignment(unitArray, this.ModificationDateTime, this.ModificationTechId);

				dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtOrderedUnit, this._lastUpdateFunctionId));
				sprocArrayList.Add(SPROC.ReleaseOrderedUnitsByBloodUnit.StoredProcName);

				//Update the QuarantineIndicator
				System.Data.DataTable dtUpdateBUS = DAL.BloodUnitStatus.GetEmptyBloodUnitStatusTableSchema(true);
				System.Data.DataTable dtPoolBUS = DAL.BloodUnitStatus.GetBloodUnitStatusByGuid(this.PooledUnit.NewUnit.BloodUnitGuid);

				System.Data.DataRow drBUS = this.PooledUnit.NewUnit.BloodUnitStatus.LoadDataRowFromThis(dtUpdateBUS.NewRow());
				drBUS[TABLE.BloodUnitStatus.UnitStatusDate] = this.ModificationDateTime;
				drBUS[TABLE.BloodUnitStatus.UnitStatusComments] = "Label Verification Failed";
				drBUS[TABLE.BloodUnitStatus.QuarantineIndicator] = true;

				dtUpdateBUS.Rows.Add(drBUS);

				dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtUpdateBUS, this._lastUpdateFunctionId));
				sprocArrayList.Add(SPROC.UpdateBloodUnitStatus.StoredProcName);
			}

			//Regardless of what's happened above, we want to release all of our pooling unit patient assignments.
			//The pooled unit, should either have it's assignment create in dtOrderedUnit, or no assignment
			System.Data.DataTable dtUpdateOU = BOL.OrderedUnit.GetEmptyOrderedUnitDataTable();

			foreach(BOL.BloodUnit poolingUnit in this.PoolingUnits)
			{
				System.Data.DataTable dtAssignments = BOL.OrderedUnit.GetOrderedUnit(poolingUnit.BloodUnitGuid, Common.LogonUser.LogonUserDivisionCode );

				foreach(System.Data.DataRow dr in dtAssignments.Rows)
				{
					System.Data.DataRow drRelease = dtUpdateOU.NewRow();
					drRelease[TABLE.OrderedUnit.BloodUnitGuid] = poolingUnit.BloodUnitGuid;
					drRelease[TABLE.OrderedUnit.ReleaseTechId] = this.ModificationTechId;
					drRelease[TABLE.OrderedUnit.ReleaseDate] = this.ModificationDateTime;
					drRelease[TABLE.OrderedUnit.LastUpdateUser] = Common.LogonUser.LogonUserName;
					drRelease[TABLE.OrderedUnit.RowVersion] = dr[TABLE.OrderedUnit.RowVersion];
					drRelease[TABLE.OrderedUnit.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					dtUpdateOU.Rows.Add(drRelease);
				}
			}

			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtUpdateOU, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.ReleaseOrderedUnitsByBloodUnit.StoredProcName);


			//*	spInsertBloodUnitModification**********************************
			System.Data.DataTable dtInsertBUM = DAL.BloodUnitModification.GetEmptyBloodUnitModificationTable(false);
			System.Guid []  bloodUnitGuids = new System.Guid [this._poolingUnits.Count];
			for (int ndx = 0; ndx < this._poolingUnits.Count; ndx++)
			{
				BOL.BloodUnit tmpUnit = (BOL.BloodUnit)this._poolingUnits[ndx];

				if (!this.IsOriginalUnit(tmpUnit.BloodUnitGuid))
				{
					System.Data.DataRow drModifiedUnit = this.LoadBloodUnitModificationInformation(this._pooledUnit, dtInsertBUM.NewRow(), tmpUnit);				

					drModifiedUnit[TABLE.BloodUnitModification.FromBloodUnitGuid] = tmpUnit.BloodUnitGuid;
					drModifiedUnit[TABLE.BloodUnitModification.OriginalUnitIndicator] = false;

					//Build our arraylist of BloodUnitModificationGuids, which we'll need for adding supplies
					//Pooling creates one Modified Unit from many origs, so we have to generate these OTF
					System.Guid newGuid = System.Guid.NewGuid();
					bumGuids.Add(newGuid);
					drModifiedUnit[TABLE.BloodUnitModification.BloodUnitModificationGuid] = newGuid;
				
					dtInsertBUM.Rows.Add(drModifiedUnit);
					bloodUnitGuids[ndx] = tmpUnit.BloodUnitGuid;
				}
				else
				{
					//Original Unit...No processing necessary.. Updates handled below
				}
			}

			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtInsertBUM, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertBloodUnitModification.StoredProcName);

			//If we've added new units, attached all the supplies/equipment
			if (bumGuids.Count > 0)
			{
				//*	spInsertModificationSupply*************************************for the new units
				System.Data.DataTable dtModificationSupply = this._pooledUnit.LoadModificationSupplyInformation(bumGuids, false);
				
				dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtModificationSupply, this._lastUpdateFunctionId));
				sprocArrayList.Add(SPROC.InsertModificationSupply.StoredProcName);

				//*	spInsertModificationEquipment**********************************
				System.Data.DataTable dtModificationEquipment = this._pooledUnit.LoadModificationEquipmentInformation(bumGuids, false);
				
				dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtModificationEquipment, this._lastUpdateFunctionId));
				sprocArrayList.Add(SPROC.InsertModificationEquipment.StoredProcName);
			}

			//*spUpdateBloodUnitModification***********************************
			System.Collections.ArrayList removedBumGuids = new ArrayList();		//Modifications that were removed
			System.Collections.ArrayList stayedBumGuids = new ArrayList();		//Modfiications that stayed within pool

			System.Data.DataTable dtUpdateBUM = DAL.BloodUnitModification.GetEmptyBloodUnitModificationTable(true);
			foreach(System.Guid originalGuid in this._originalPoolingUnits)
			{
				foreach(System.Data.DataRow drOrigs in this._dtOriginalPoolingUnits.Rows)
				{
					//Find the current original unit in our datatable and process it
					if ((System.Guid)drOrigs[TABLE.BloodUnitModification.FromBloodUnitGuid] == originalGuid)
					{
						System.Data.DataRow drModifiedUnit = dtUpdateBUM.NewRow();

						drModifiedUnit[TABLE.BloodUnitModification.BloodUnitModificationGuid] = drOrigs[TABLE.BloodUnitModification.BloodUnitModificationGuid];
						drModifiedUnit[TABLE.BloodUnitModification.ModificationDate] = drOrigs[TABLE.BloodUnitModification.ModificationDate];
						drModifiedUnit[TABLE.BloodUnitModification.ModificationTechId] = drOrigs[TABLE.BloodUnitModification.ModificationTechId];
						
						if (this._pooledUnit.NewLabelVerificationDateTime != System.DateTime.MinValue)
							drModifiedUnit[TABLE.BloodUnitModification.NewLabelVerificationDatetime] = this._pooledUnit.NewLabelVerificationDateTime;
						
						drModifiedUnit[TABLE.BloodUnitModification.DivisionCode] = drOrigs[TABLE.BloodUnitModification.DivisionCode];
						drModifiedUnit[TABLE.BloodUnitModification.RowVersion] = drOrigs[TABLE.BloodUnitModification.RowVersion];
						drModifiedUnit[TABLE.BloodUnitModification.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
						drModifiedUnit[TABLE.BloodUnitModification.Comments] = this.Comments;
						drModifiedUnit[TABLE.BloodUnitModification.OriginalUnitIndicator] = drOrigs[TABLE.BloodUnitModification.OriginalUnitIndicator];
						drModifiedUnit[TABLE.BloodUnitModification.ModificationCost] = drOrigs[TABLE.BloodUnitModification.ModificationCost];

						//If this is a removed Unit, set the inactive RecordStatusCode, leave LabelVerify
						if (!this._remainingOriginalUnits.Contains(originalGuid))
						{
							drModifiedUnit[TABLE.BloodUnitModification.NewLabelVerificationDatetime] = drOrigs[TABLE.BloodUnitModification.NewLabelVerificationDatetime];
							drModifiedUnit[TABLE.BloodUnitModification.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive);
							removedBumGuids.Add((System.Guid)drOrigs[TABLE.BloodUnitModification.BloodUnitModificationGuid]);
						}
						else
						{
							//This unit stayed from the original Edit or are new
							stayedBumGuids.Add((System.Guid)drOrigs[TABLE.BloodUnitModification.BloodUnitModificationGuid]);
						}

						drModifiedUnit[TABLE.BloodUnitModification.PoolEditedIndicator] = true;

						dtUpdateBUM.Rows.Add(drModifiedUnit);
						break;
					}
				}
			}

			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtUpdateBUM, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.UpdateBloodUnitModification.StoredProcName);

			//"Updated" the modified target unit
			System.Data.DataTable dtUpdateBU = DAL.BloodUnit.GetEmptyBloodUnitTableSchema(true);
			dtUpdateBU.Rows.Add(this.LoadTargetBloodUnitInformation(this.PooledUnit, dtUpdateBU.NewRow()));
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtUpdateBU, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.UpdateBloodUnit.StoredProcName);				
			
			//Save off the new expiration date
			System.Data.DataTable dtUpdateBUMedia = DAL.BloodUnitMedia.GetEmptyBloodUnitMediaTableSchema(true);
			dtUpdateBUMedia.Rows.Add(this.LoadTargetBloodUnitInformation(this.PooledUnit, dtUpdateBUMedia.NewRow()));
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtUpdateBUMedia, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.UpdateBloodUnitMedia.StoredProcName);		

			//Updating the BloodUnitFinance with the new UnitCost -- as a result of the Modification Charge
			System.Data.DataTable dtUpdateBUFinance = DAL.BloodUnitFinance.GetEmptyBloodUnitFinanceTableSchema(true);
			dtUpdateBUFinance.Rows.Add(this.PooledUnit.LoadBloodUnitInformation(dtUpdateBUFinance.NewRow()));
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtUpdateBUFinance, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.UpdateBloodUnitFinance.StoredProcName);	

			//If Units were removed, deactivate all their equipment and supplies -- might be overkill, since the bum is inactivated
			System.Data.DataTable dtEquipUpdate = DAL.BloodUnitModification.GetEmptyModificationEquipmentTable(true);
			System.Data.DataTable dtEquipInsert = DAL.BloodUnitModification.GetEmptyModificationEquipmentTable(false);
			System.Data.DataTable dtSupplyInsert = DAL.BloodUnitModification.GetEmptyModificationSupplyTable(false);
			System.Data.DataTable dtSupplyUpdate = DAL.BloodUnitModification.GetEmptyModificationSupplyTable(true);


			if (removedBumGuids.Count > 0)
			{
				foreach(System.Guid bumGuid in removedBumGuids)
				{
					System.Data.DataTable dtKillSupplies = DAL.BloodUnitModification.GetModificationSupplies(bumGuid);
					System.Data.DataTable dtKillEquipment = DAL.BloodUnitModification.GetModificationEquipment(bumGuid);

					//Update any equipment
					foreach(System.Data.DataRow dr in dtKillEquipment.Rows)
					{
						System.Data.DataRow drUpdEquip = dtEquipUpdate.NewRow();
						drUpdEquip[TABLE.ModificationEquipment.ModificationEquipmentGuid] = dr[TABLE.ModificationEquipment.ModificationEquipmentGuid];
						drUpdEquip[TABLE.ModificationEquipment.BloodUnitModificationGuid] = dr[TABLE.ModificationEquipment.BloodUnitModificationGuid];
						drUpdEquip[TABLE.ModificationEquipment.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive);
						drUpdEquip[TABLE.ModificationEquipment.RowVersion] = dr[TABLE.ModificationEquipment.RowVersion];
						dtEquipUpdate.Rows.Add(drUpdEquip);
					}
				
					//Update any supplies
					foreach(System.Data.DataRow dr in dtKillSupplies.Rows)
					{
						System.Data.DataRow drUpdSupply = dtSupplyUpdate.NewRow();
						drUpdSupply[TABLE.ModificationSupply.ModificationSupplyGuid] = dr[TABLE.ModificationSupply.ModificationSupplyGuid];
						drUpdSupply[TABLE.ModificationSupply.BloodUnitModificationGuid] = dr[TABLE.ModificationSupply.BloodUnitModificationGuid];
						drUpdSupply[TABLE.ModificationSupply.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive);
						drUpdSupply[TABLE.ModificationSupply.RowVersion] = dr[TABLE.ModificationSupply.RowVersion];
						dtSupplyUpdate.Rows.Add(drUpdSupply);
					}
				}
			}

			//If units were left as-is, in the modification, we'll deactivate any supplies/equip that were removed And add any new stuff
			if (stayedBumGuids.Count > 0)
			{
				System.Collections.ArrayList tmpArray = new ArrayList();
				tmpArray.Add(stayedBumGuids[0]);

				//Get the supplies --  This will return us one of everything that's currently attached, so we don't have to worry about containers/scd checking
				System.Data.DataTable dtCurrentSupplies = this._pooledUnit.LoadModificationSupplyInformation(tmpArray, false);

				foreach(System.Guid bumGuid in stayedBumGuids)
				{
					//Find out if any equipment was removed
					foreach(System.Data.DataRow drOrigEquip in this._dtOriginalPoolEquipment.Rows)
					{
						bool foundEquip = false;

						foreach(BOL.Equipment tmpEquip in this.PooledUnit.EquipmentList)
						{
							if ((System.Guid)drOrigEquip[TABLE.ModificationEquipment.EquipmentGuid] == tmpEquip.EquipmentGuid)
							{
								foundEquip = true;
								break;
							}
						}

						if (!foundEquip)		//Equipment was removed, update the MOdificationequipment record
						{
							System.Data.DataTable dtKillEquipment = DAL.BloodUnitModification.GetModificationEquipment(bumGuid);

							foreach(System.Data.DataRow drTmp in dtKillEquipment.Rows)
							{
								if ((System.Guid)drOrigEquip[TABLE.ModificationEquipment.EquipmentGuid] == (System.Guid)drTmp[TABLE.ModificationEquipment.EquipmentGuid])
								{
									System.Data.DataRow drUpdEquip = dtEquipUpdate.NewRow();
									drUpdEquip[TABLE.ModificationEquipment.ModificationEquipmentGuid] = drTmp[TABLE.ModificationEquipment.ModificationEquipmentGuid];
									drUpdEquip[TABLE.ModificationEquipment.BloodUnitModificationGuid] = drTmp[TABLE.ModificationEquipment.BloodUnitModificationGuid];
									drUpdEquip[TABLE.ModificationEquipment.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive);
									drUpdEquip[TABLE.ModificationEquipment.RowVersion] = drTmp[TABLE.ModificationEquipment.RowVersion];
									dtEquipUpdate.Rows.Add(drUpdEquip);
								}
							}
						}

					}//foreach(System.Data....

					//Now, reverse it, and find out if any equipment was added
					foreach(BOL.Equipment tmpEquip in this.PooledUnit.EquipmentList)
					{
						bool newEquip = true;
						foreach(System.Data.DataRow drOrigEquip in this._dtOriginalPoolEquipment.Rows)
						{
							if (tmpEquip.EquipmentGuid == (System.Guid)drOrigEquip[TABLE.ModificationEquipment.EquipmentGuid])
							{
								newEquip = false;
								break;
							}
						}

						if (newEquip)
						{
							System.Data.DataRow drInsEquip = dtEquipInsert.NewRow();
							drInsEquip[TABLE.ModificationEquipment.ModificationEquipmentGuid] = System.Guid.NewGuid();
							drInsEquip[TABLE.ModificationEquipment.BloodUnitModificationGuid] = bumGuid;
							drInsEquip[TABLE.ModificationEquipment.EquipmentGuid] = tmpEquip.EquipmentGuid;
							drInsEquip[TABLE.ModificationEquipment.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
							dtEquipInsert.Rows.Add(drInsEquip);
						}
					}//foreach(BOL.Equip ...

					//Find out if any supplies was removed
					foreach(System.Data.DataRow drOrigSupply in this._dtOriginalPoolSupply.Rows)
					{
						bool foundSupply = false;

						foreach(System.Data.DataRow drCurrSupply in dtCurrentSupplies.Rows)
						{
							if ((System.Guid)drOrigSupply[TABLE.ModificationSupply.SupplyGuid] == (System.Guid)drCurrSupply[TABLE.ModificationSupply.SupplyGuid])
							{
								foundSupply = true;
								break;
							}
						}

						if (!foundSupply)		//Supply was removed, update the MOdificationSupply record
						{
							System.Data.DataTable dtKillSupplies = DAL.BloodUnitModification.GetModificationSupplies(bumGuid);

							foreach(System.Data.DataRow drTmp in dtKillSupplies.Rows)
							{
								if ((System.Guid)drTmp[TABLE.ModificationSupply.SupplyGuid] == (System.Guid)drOrigSupply[TABLE.ModificationSupply.SupplyGuid])
								{
									System.Data.DataRow drUpdSupply = dtSupplyUpdate.NewRow();
									drUpdSupply[TABLE.ModificationSupply.ModificationSupplyGuid] = drTmp[TABLE.ModificationSupply.ModificationSupplyGuid];
									drUpdSupply[TABLE.ModificationSupply.BloodUnitModificationGuid] = drTmp[TABLE.ModificationSupply.BloodUnitModificationGuid];
									drUpdSupply[TABLE.ModificationSupply.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive);
									drUpdSupply[TABLE.ModificationSupply.RowVersion] = drTmp[TABLE.ModificationSupply.RowVersion];
									dtSupplyUpdate.Rows.Add(drUpdSupply);
								}
							}
						}

					}//foreach(System.Data....

					//Now, reverse it, and find out if any supply was added
					foreach(System.Data.DataRow drCurrSupply in dtCurrentSupplies.Rows)
					{
						bool newSupply = true;
						foreach(System.Data.DataRow drOrigSupply in this._dtOriginalPoolSupply.Rows)
						{
							if ((System.Guid)drCurrSupply[TABLE.ModificationSupply.SupplyGuid] == (System.Guid)drOrigSupply[TABLE.ModificationSupply.SupplyGuid])
							{
								newSupply = false;
								break;
							}
						}

						if (newSupply)
						{
							System.Data.DataRow drInsSupply = dtSupplyInsert.NewRow();
							drInsSupply[TABLE.ModificationSupply.ModificationSupplyGuid] = System.Guid.NewGuid();
							drInsSupply[TABLE.ModificationSupply.BloodUnitModificationGuid] = bumGuid;
							drInsSupply[TABLE.ModificationSupply.SupplyGuid] = drCurrSupply[TABLE.ModificationSupply.SupplyGuid];
							drInsSupply[TABLE.ModificationSupply.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
							dtSupplyInsert.Rows.Add(drInsSupply);
						}
					}//foreach(BOL.Equip ...
				}//foreach(System.Guid..
			}//end if (stayedBumGuids.Count > 0)

			//*	spUpdateModificationEquipment*************************************
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtEquipUpdate, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.UpdateModificationEquipment.StoredProcName);

			//*	spUpdateModificationSupply*************************************
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtSupplyUpdate, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.UpdateModificationSupply.StoredProcName);
			
			//*	spInsertModificationEquipment*************************************
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtEquipInsert, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertModificationEquipment.StoredProcName);
				
			//*	spInsertModificationSupply*************************************
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtSupplyInsert, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertModificationSupply.StoredProcName);


			//Since we already created the BloodUnitTest on pool creation, we're now going to 
			//just edit the record -- if an existing record is found
			string testResult = (this.TargetCmvNeg) ? "N" : "P";
			System.Data.DataTable dtCMVCurrent = BOL.BloodUnitTest.GetBloodUnitTestResult(this.PooledUnit.ToBloodUnitGuid, (int)Common.TestType.CMV, Common.LogonUser.LogonUserDivisionCode);
			if (dtCMVCurrent.Rows.Count == 0)
			{
				//* spInsertBloodUnitTest*********************************************
				BOL.BloodUnitTest unitTest = new BOL.BloodUnitTest(System.Guid.NewGuid(), this._pooledUnit.ToBloodUnitGuid, (int)Common.TestType.CMV, VBECSDateTime.GetDivisionCurrentDateTime(), testResult, string.Empty, "M", Common.LogonUser.LogonUserDivisionCode);
				System.Data.DataTable dtBloodUnitTest = DAL.BloodUnitTest.GetEmptyBloodUnitTestTableSchema(false);
				dtBloodUnitTest.Rows.Add(unitTest.LoadDataRowFromThis(dtBloodUnitTest.NewRow()));

				dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtBloodUnitTest, this._lastUpdateFunctionId));
				sprocArrayList.Add(SPROC.InsertBloodUnitTest.StoredProcName);
			}
			else
			{
				BOL.BloodUnitTest tmpBUT = new BOL.BloodUnitTest(dtCMVCurrent.Rows[0]);
				if (tmpBUT.TestResultId.ToUpper() != testResult)
				{
					System.Data.DataTable dtUpdateBUT = DAL.BloodUnitModification.GetEmptyBloodUnitTestTable();
					dtUpdateBUT.Rows.Add(tmpBUT.LoadDataRowFromThis(dtUpdateBUT.NewRow()));
					
					if (dtUpdateBUT.Rows[0].IsNull(TABLE.BloodUnitTest.TestTechId))
					{
						dtUpdateBUT.Rows[0][TABLE.BloodUnitTest.TestTechId] = Common.LogonUser.LogonUserName;
					}
					if (dtUpdateBUT.Rows[0].IsNull(TABLE.BloodUnitTest.EntryTechId))
					{
						dtUpdateBUT.Rows[0][TABLE.BloodUnitTest.EntryTechId] = Common.LogonUser.LogonUserName;
					}

					dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtUpdateBUT, this._lastUpdateFunctionId));
					sprocArrayList.Add(SPROC.UpdateBloodUnitTestResult.StoredProcName);
				}
			}



			
			//Any exceptions generated by supplies  -- spInsertExceptionModification****************
			dtArrayList.Add(this.GetSupplyExceptionsDataTable(this.PooledUnit.Container));
			sprocArrayList.Add(SPROC.InsertExceptionUnitModification.StoredProcName);

			foreach(BOL.Supply tmpSupply in this.PooledUnit.SupplyList)
			{
				dtArrayList.Add(this.GetSupplyExceptionsDataTable(tmpSupply));
				sprocArrayList.Add(SPROC.InsertExceptionUnitModification.StoredProcName);
			}

			dtArrayList.Add(this.GetSupplyExceptionsDataTable(this.PooledUnit.SCDWafer));
			sprocArrayList.Add(SPROC.InsertExceptionUnitModification.StoredProcName);


			//*spInsertExceptionModification********************************************
			dtArrayList.Add(this.GetExceptionsDataTable(this.Exceptions));
			sprocArrayList.Add(SPROC.InsertExceptionUnitModification.StoredProcName);

			dtArrayList.Add(this.GetExceptionsDataTable(this.LabelExceptions));
			sprocArrayList.Add(SPROC.InsertExceptionLabelMismatch.StoredProcName);
			

			//BR_24.06 - Calculate workload
			//* spInsertWorkloadEvents*****************************************
			System.Data.DataTable dtWorkload = BOL.WorkloadEvent.GenerateWorkloadData(this.WorkloadCode, bloodUnitGuids, Common.WorkloadTransactionType.Unit, false);
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtWorkload, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertWorkloadEvents.StoredProcName);

			DataTable dtBCEMessages = null;
			DataTable dtTempBCEMessages = null;

			//CR 2941
			//CR 2945
			//generate table with BCE_MESSAGE
			//The Unit(s) are added to a pool, send BCE message releasing all unit assignments

			//getting all patients assigned to a unit
			foreach(BOL.BloodUnit poolingUnit in this.PoolingUnits)
			{
				DataTable dtPatients = DAL.PatientAssignments.GetPatientsAssignmentDetails(poolingUnit.BloodUnitGuid);
				Guid assignedPatientGuid = Guid.Empty;

				for (int j=0; j< dtPatients.Rows.Count; j++)
				{
					assignedPatientGuid = (Guid)dtPatients.Rows[j][Common.VbecsTables.Patient.PatientGuid];

					//CR 3015
					if (dtBCEMessages == null)
					{
						dtBCEMessages = OrderedUnit.UpdateBceForRelease(poolingUnit.BloodUnitGuid, assignedPatientGuid, false);
					}
					else
					{
						dtTempBCEMessages = OrderedUnit.UpdateBceForRelease(poolingUnit.BloodUnitGuid, assignedPatientGuid, false);
						if (dtTempBCEMessages.Rows.Count > 0)
							dtBCEMessages.Rows.Add(dtTempBCEMessages.Rows[0].ItemArray);
					}
				}
			}

			if (dtBCEMessages != null)
			{
				if (dtBCEMessages.Rows.Count > 0)
				{
					dtArrayList.Add(dtBCEMessages);
					sprocArrayList.Add(SPROC.HL7InsertMessageLog.StoredProcName);
				}
			}

			//Grab the OrderedUnit information before it is released -- the GetOrderedUnit method cannot find it otherwise
			System.Data.DataTable dtReleasingAssignments = null;
			System.Guid patientGuid = Guid.Empty;

			if (this.PooledUnit.LabelVerificationFailed == false)
			{
				if (this._inheritedPatientAssignment[0] != System.Guid.Empty)
				{
					patientGuid = this._inheritedPatientAssignment[1];
					//CR 3015
					dtReleasingAssignments = BOL.OrderedUnit.GetOrderedUnit(this._inheritedPatientAssignment[0], Common.LogonUser.LogonUserDivisionCode );
				}
			}
			//BCE_MESSAGE end

			
			int result = (new Common.StoredProcedure().TransactionalGetValue(sprocArrayList, dtArrayList));		
			//send BCE message reserving a new unit only if the user decided to move unit assignment during modification and 
			//if the label verification did not fail
			
			if (result == 0)
			{
				if (this.PooledUnit.LabelVerificationFailed == false)
				{
					if (this._inheritedPatientAssignment[0] != System.Guid.Empty)
					{
						Guid orderedComponentGuid = Guid.Empty;
						var drAssignments = DrAssignments(dtReleasingAssignments, patientGuid);
						if (drAssignments.Length > 0)
							orderedComponentGuid = (Guid)drAssignments[0][Common.VbecsTables.OrderedUnit.OrderedComponentGuid];

						OrderedUnit.UpdateAndSaveBce(this.PooledUnit.ToBloodUnitGuid, patientGuid, orderedComponentGuid, "", Common.HL7BPDispenseStatus.RS);
					}
				}
			}

			return result;
		}

	    private static DataRow[] DrAssignments(DataTable dtReleasingAssignments, Guid patientGuid)
	    {
           DataRow[] drAssignments =
	           dtReleasingAssignments.Select(Common.VbecsTables.Patient.PatientGuid + " = '" + patientGuid.ToString() + "'");
	       return drAssignments;
	        
	    }

	    /// <summary>
		/// Gathers and saves information related to a pool
		/// Implements BR_24.24, BR_24.06
		/// </summary>
		/// <returns></returns>
		private int PrepareAndSavePool()
		{
			System.Guid shipmentGuid = System.Guid.NewGuid();
			System.Guid localSupplierGuid = this.PooledUnit.NewUnit.IncomingShipment.LocalSupplierGuid;
			System.Guid targetBPGuid = this.PooledUnit.TargetBloodProduct.BloodProductGuid;

			System.Collections.ArrayList bumGuids = new System.Collections.ArrayList();
			System.Collections.ArrayList dtArrayList = new System.Collections.ArrayList();
			System.Collections.ArrayList sprocArrayList = new System.Collections.ArrayList();
			
			//*	spInsertShipment***********************************************
			System.Data.DataTable dtShipment = DAL.BloodUnitModification.GetEmptyShipmentTable();
			dtShipment.Rows.Add(this.LoadShipmentInformation(dtShipment.NewRow(), shipmentGuid, localSupplierGuid, targetBPGuid));
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtShipment,this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertShipment.StoredProcName);

			//*	spInsertMaintainableBloodUnit**********************************
			System.Data.DataTable dtBloodUnit = DAL.BloodUnit.GetEmptyBloodUnitTableSchema(false);
			System.Data.DataTable dtBloodUnitFinance = DAL.BloodUnitFinance.GetEmptyBloodUnitFinanceTableSchema(false);
			System.Data.DataTable dtBloodUnitMedia = DAL.BloodUnitMedia.GetEmptyBloodUnitMediaTableSchema(false);
			System.Data.DataTable dtBloodUnitStatus = DAL.BloodUnitStatus.GetEmptyBloodUnitStatusTableSchema(false);

			dtBloodUnit.Rows.Add(this.PooledUnit.LoadBloodUnitInformation(dtBloodUnit.NewRow()));
			dtBloodUnitFinance.Rows.Add(this.PooledUnit.LoadBloodUnitInformation(dtBloodUnitFinance.NewRow()));
			dtBloodUnitMedia.Rows.Add(this.PooledUnit.LoadBloodUnitInformation(dtBloodUnitMedia.NewRow()));
			dtBloodUnitStatus.Rows.Add(this.PooledUnit.LoadBloodUnitInformation(dtBloodUnitStatus.NewRow()));

			//spInsertBloodUnit, spInsertBloodUnitFinance, spInsertBloodUnitMedia, spInsertBloodUnitStatus*********************
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtBloodUnit, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertBloodUnit.StoredProcName);
			
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtBloodUnitFinance, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertBloodUnitFinance.StoredProcName);
			
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtBloodUnitMedia, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertBloodUnitMedia.StoredProcName);

			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtBloodUnitStatus, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertBloodUnitStatus.StoredProcName);
			//******************************************************************************************************************


			//BR_26.29 - Dont' save patient assignments if the verification failed
			if (!this.PooledUnit.LabelVerificationFailed)
			{
				//spInsertOrderedUnit - creates patient assignments for the new unit
				System.Data.DataTable dtOrderedUnit = DAL.BloodUnitModification.GetEmptyOrderedUnitTable();

				if (this._inheritedPatientAssignment[0] != System.Guid.Empty)
				{
					BOL.OrderedUnit tmpOU = new BOL.OrderedUnit(this._inheritedPatientAssignment[0], this._inheritedPatientAssignment[1]);
					tmpOU.BloodUnitGuid = this.PooledUnit.ToBloodUnitGuid;	//Assign the new unit to the patient
					tmpOU.OrderedUnitGuid = System.Guid.NewGuid();	//Sproc accepts this
					if (tmpOU.CurrentCrossmatchDate == System.DateTime.MinValue)		//LoadDataRowFromThis defaults to MINVALUE if NULL
					{
						if (dtOrderedUnit.Columns.Contains(TABLE.OrderedUnit.CrossmatchDate))
							dtOrderedUnit.Columns.Remove(TABLE.OrderedUnit.CrossmatchDate);
						if (dtOrderedUnit.Columns.Contains(TABLE.OrderedUnit.CrossmatchResultCode))
							dtOrderedUnit.Columns.Remove(TABLE.OrderedUnit.CrossmatchResultCode);
						if (dtOrderedUnit.Columns.Contains(TABLE.OrderedUnit.CrossmatchTechId))
							dtOrderedUnit.Columns.Remove(TABLE.OrderedUnit.CrossmatchTechId);
					}
					dtOrderedUnit.Rows.Add(tmpOU.LoadDataRowFromThis(dtOrderedUnit.NewRow()));
				}

				dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtOrderedUnit, this._lastUpdateFunctionId));
				sprocArrayList.Add(SPROC.InsertOrderedUnit.StoredProcName);
			}


			//Regardless of what's happened above, we want to release all of our pooling unit patient assignments.
			//The pooled unit, should either have it's assignment create in dtOrderedUnit, or no assignment
			System.Data.DataTable dtUpdateOU = BOL.OrderedUnit.GetEmptyOrderedUnitDataTable();

			foreach(BOL.BloodUnit poolingUnit in this.PoolingUnits)
			{
				System.Data.DataTable dtAssignments = BOL.OrderedUnit.GetOrderedUnit(poolingUnit.BloodUnitGuid, Common.LogonUser.LogonUserDivisionCode );

				foreach(System.Data.DataRow dr in dtAssignments.Rows)
				{
					System.Data.DataRow drRelease = dtUpdateOU.NewRow();
					drRelease[TABLE.OrderedUnit.BloodUnitGuid] = poolingUnit.BloodUnitGuid;
					drRelease[TABLE.OrderedUnit.ReleaseTechId] = this.ModificationTechId;
					drRelease[TABLE.OrderedUnit.ReleaseDate] = this.ModificationDateTime;
					drRelease[TABLE.OrderedUnit.LastUpdateUser] = Common.LogonUser.LogonUserName;
					drRelease[TABLE.OrderedUnit.RowVersion] = dr[TABLE.OrderedUnit.RowVersion];
					drRelease[TABLE.OrderedUnit.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					dtUpdateOU.Rows.Add(drRelease);
				}
			}

			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtUpdateOU, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.ReleaseOrderedUnitsByBloodUnit.StoredProcName);


			//* spInsertBloodUnitShipment
			System.Data.DataTable dtBloodUnitShipment = DAL.BloodUnitModification.GetEmptyBloodUnitShipmentTable();
			dtBloodUnitShipment.Rows.Add(this.PooledUnit.LoadBloodUnitShipmentInformation(dtBloodUnitShipment.NewRow(), shipmentGuid));

			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtBloodUnitShipment, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertBloodUnitShipment.StoredProcName);

			//*	spInsertBloodUnitModification**********************************
			System.Data.DataTable dtBloodUnitModification = DAL.BloodUnitModification.GetEmptyBloodUnitModificationTable(false);
			System.Guid []  bloodUnitGuids = new System.Guid [this._poolingUnits.Count];
			int ndx=0;
			foreach(BOL.BloodUnit tmpSrcUnit in this._poolingUnits)
			{
				System.Data.DataRow drModifiedUnit = this.LoadBloodUnitModificationInformation(this._pooledUnit, dtBloodUnitModification.NewRow(), tmpSrcUnit);				

				drModifiedUnit[TABLE.BloodUnitModification.FromBloodUnitGuid] = tmpSrcUnit.BloodUnitGuid;

				//Build our arraylist of BloodUnitModificationGuids, which we'll need for adding supplies
				//Pooling creates one Modified Unit from many normal, so we have to generate these OTF
				bumGuids.Add(System.Guid.NewGuid());
				drModifiedUnit[TABLE.BloodUnitModification.BloodUnitModificationGuid] = (System.Guid)bumGuids[ndx];
				
				dtBloodUnitModification.Rows.Add(drModifiedUnit);
				bloodUnitGuids[ndx] = tmpSrcUnit.BloodUnitGuid;
				ndx++;
			}

			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtBloodUnitModification, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertBloodUnitModification.StoredProcName);

			//*	spInsertModificationSupply*************************************
			System.Data.DataTable dtModificationSupply = this._pooledUnit.LoadModificationSupplyInformation(bumGuids,false);
				
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtModificationSupply, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertModificationSupply.StoredProcName);

			//*	spInsertModificationEquipment**********************************
			System.Data.DataTable dtModificationEquipment = this._pooledUnit.LoadModificationEquipmentInformation(bumGuids,false);
				
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtModificationEquipment, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertModificationEquipment.StoredProcName);

			//* spInsertBloodUnitTest******************************************
			string testResult = (this.TargetCmvNeg) ? "N" : "P";

			BOL.BloodUnitTest unitTest = new BOL.BloodUnitTest(System.Guid.NewGuid(), this._pooledUnit.ToBloodUnitGuid, (int)Common.TestType.CMV, VBECSDateTime.GetDivisionCurrentDateTime(), testResult, string.Empty, "M", Common.LogonUser.LogonUserDivisionCode);
			System.Data.DataTable dtBloodUnitTest = DAL.BloodUnitTest.GetEmptyBloodUnitTestTableSchema(false);
			dtBloodUnitTest.Rows.Add(unitTest.LoadDataRowFromThis(dtBloodUnitTest.NewRow()));

			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtBloodUnitTest, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertBloodUnitTest.StoredProcName);
			
			//Any exceptions generated by supplies  -- spInsertExceptionModification****************
			dtArrayList.Add(this.GetSupplyExceptionsDataTable(this.PooledUnit.Container));
			sprocArrayList.Add(SPROC.InsertExceptionUnitModification.StoredProcName);

			foreach (BOL.Supply tmpSupply in this.PooledUnit.SupplyList)
			{
				dtArrayList.Add(this.GetSupplyExceptionsDataTable(tmpSupply));
				sprocArrayList.Add(SPROC.InsertExceptionUnitModification.StoredProcName);
			}

			dtArrayList.Add(this.GetSupplyExceptionsDataTable(this.PooledUnit.SCDWafer));
			sprocArrayList.Add(SPROC.InsertExceptionUnitModification.StoredProcName);


			//*spInsertExceptionModification********************************************
			dtArrayList.Add(this.GetExceptionsDataTable(this.Exceptions));
			sprocArrayList.Add(SPROC.InsertExceptionUnitModification.StoredProcName);

			dtArrayList.Add(this.GetExceptionsDataTable(this.LabelExceptions));
			sprocArrayList.Add(SPROC.InsertExceptionLabelMismatch.StoredProcName);
			
			
			//BR_24.06 - Calculate workload for Pool/Thaw - handled here
			//* spInsertWorkloadEvents*****************************************
			//CR1982 - Workload should only apply to the created pool
			System.Guid[] workloadGuid = {this.PooledUnit.ToBloodUnitGuid};
			System.Data.DataTable dtWorkload = BOL.WorkloadEvent.GenerateWorkloadData(this.WorkloadCode, workloadGuid, Common.WorkloadTransactionType.Unit, false);
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtWorkload, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertWorkloadEvents.StoredProcName);

			if (this.PooledUnit.NewUnit.EyeReadableUnitId == this._calculatedPooledUnitId)
			{
				//Inactivate the PoolCounter - updated in CR3320 to use DataRow instead of DataTable  //CR3404 added next 2 lines back- dumb Lohse
                if (this._drUsedPoolCounter.Table.Columns.Contains(TABLE.PoolCounter.CounterValue)) this._drUsedPoolCounter.Table.Columns.Remove(TABLE.PoolCounter.CounterValue);
                if (this._drUsedPoolCounter.Table.Columns.Contains(TABLE.VbecsLock.VbecsLockGuid)) this._drUsedPoolCounter.Table.Columns.Remove(TABLE.VbecsLock.VbecsLockGuid);
                this._drUsedPoolCounter[TABLE.PoolCounter.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive);
			    
                this._drUsedPoolCounter.Table.Columns.Add(TABLE.PoolCounter.LastUpdateUser);
                this._drUsedPoolCounter[TABLE.PoolCounter.LastUpdateUser] = Common.LogonUser.LogonUserName;
                dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(this._drUsedPoolCounter.Table, this._lastUpdateFunctionId));

				sprocArrayList.Add(SPROC.UpdatePoolCounter.StoredProcName);
			}

			DataTable dtBCEMessages = null;
			DataTable dtTempBCEMessages = null;

			//CR 2941
			//CR 2945
			//generate table with BCE_MESSAGE
			//The Unit(s) are pooled, send BCE message releasing all unit assignments

			//getting all patients assigned to a unit
			foreach(BOL.BloodUnit poolingUnit in this.PoolingUnits)
			{
				DataTable dtPatients = DAL.PatientAssignments.GetPatientsAssignmentDetails(poolingUnit.BloodUnitGuid);
				Guid assignedPatientGuid = Guid.Empty;

				for (int j=0; j< dtPatients.Rows.Count; j++)
				{
					assignedPatientGuid = (Guid)dtPatients.Rows[j][Common.VbecsTables.Patient.PatientGuid];
					
					//CR 3015
					if (dtBCEMessages == null)
					{
						dtBCEMessages = OrderedUnit.UpdateBceForRelease(poolingUnit.BloodUnitGuid, assignedPatientGuid, false);
					}
					else
					{
						dtTempBCEMessages = OrderedUnit.UpdateBceForRelease(poolingUnit.BloodUnitGuid, assignedPatientGuid, false);
						if (dtTempBCEMessages.Rows.Count > 0)
							dtBCEMessages.Rows.Add(dtTempBCEMessages.Rows[0].ItemArray);
					}
				}
			}

			if (dtBCEMessages != null)
			{
				if (dtBCEMessages.Rows.Count > 0)
				{
					dtArrayList.Add(dtBCEMessages);
					sprocArrayList.Add(SPROC.HL7InsertMessageLog.StoredProcName);
				}
			}

			System.Data.DataTable dtReleasingAssignments = null;
			System.Guid patientGuid = Guid.Empty;

			//Grab the OrderedUnit information before it is released -- the GetOrderedUnit method cannot find it otherwise
			if (this.PooledUnit.LabelVerificationFailed == false)
			{
				if (this._inheritedPatientAssignment[0] != System.Guid.Empty)
				{
					patientGuid = this._inheritedPatientAssignment[1];
					//CR 3015
					dtReleasingAssignments = BOL.OrderedUnit.GetOrderedUnit(this._inheritedPatientAssignment[0], Common.LogonUser.LogonUserDivisionCode );
				}
			}

			//BCE_MESSAGE end

			int result = (new Common.StoredProcedure().TransactionalGetValue(sprocArrayList, dtArrayList));

			//send BCE message reserving a new unit only if the user decided to move unit assignment during modification and 
			//if the label verification did not fail
			
			if (result == 0)
			{
				if (this.PooledUnit.LabelVerificationFailed == false)
				{
					if (this._inheritedPatientAssignment[0] != System.Guid.Empty)
					{
						Guid orderedComponentGuid = Guid.Empty;
						var drAssignments = DataRows(dtReleasingAssignments, patientGuid);
						if (drAssignments.Length > 0)
							orderedComponentGuid = (Guid)drAssignments[0][Common.VbecsTables.OrderedUnit.OrderedComponentGuid];

						OrderedUnit.UpdateAndSaveBce(this.PooledUnit.ToBloodUnitGuid, patientGuid, orderedComponentGuid, "", Common.HL7BPDispenseStatus.RS);
					}
				}
			}

			return result;
		}

	    private static DataRow[] DataRows(DataTable dtReleasingAssignments, Guid patientGuid)
	    {
	        DataRow[] drAssignments =
	            dtReleasingAssignments.Select(Common.VbecsTables.Patient.PatientGuid + " = '" + patientGuid.ToString() + "'");
	        return drAssignments;
	    }

	    /// <summary>
		/// Gathers and saves information related to a non split/pool modification
		/// Implements BR_26.23
		/// </summary>
		/// <returns></returns>
		private int PrepareAndSaveModification()
		{
			System.Collections.ArrayList dtArrayList = new System.Collections.ArrayList();
			System.Collections.ArrayList sprocArrayList = new System.Collections.ArrayList();
			System.Collections.ArrayList bumGuids = new System.Collections.ArrayList();		//Needed for consistancy with POOL
			bumGuids.Add(this.ModifiedUnit.BloodUnitModificationGuid);

			//*	spInsertBloodUnitModification**********************************
			System.Data.DataTable dtBloodUnitModification = DAL.BloodUnitModification.GetEmptyBloodUnitModificationTable(false);			
			dtBloodUnitModification.Rows.Add(this.LoadBloodUnitModificationInformation(this.ModifiedUnit, dtBloodUnitModification.NewRow(), this.OriginalUnit));

			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtBloodUnitModification, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertBloodUnitModification.StoredProcName);
			
			//"Create" the new modified target unit
			System.Data.DataTable dtUpdateBU = DAL.BloodUnit.GetEmptyBloodUnitTableSchema(true);
			dtUpdateBU.Rows.Add(this.LoadTargetBloodUnitInformation(this.ModifiedUnit, dtUpdateBU.NewRow()));
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtUpdateBU, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.UpdateBloodUnit.StoredProcName);				
			
			//Save off the new expiration date
			System.Data.DataTable dtUpdateBUMedia = DAL.BloodUnitMedia.GetEmptyBloodUnitMediaTableSchema(true);
			dtUpdateBUMedia.Rows.Add(this.LoadTargetBloodUnitInformation(this.ModifiedUnit, dtUpdateBUMedia.NewRow()));
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtUpdateBUMedia, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.UpdateBloodUnitMedia.StoredProcName);				
		
			//Updating the BloodUnitFinance with the new UnitCost -- as a result of the Modification Charge
			System.Data.DataTable dtUpdateBUFinance = DAL.BloodUnitFinance.GetEmptyBloodUnitFinanceTableSchema(true);
			dtUpdateBUFinance.Rows.Add(this.ModifiedUnit.LoadBloodUnitInformation(dtUpdateBUFinance.NewRow()));
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtUpdateBUFinance, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.UpdateBloodUnitFinance.StoredProcName);				
		
			//If the LabelVerification didn't fail, then we leave the assignments as is...
			if (this.ModifiedUnit.LabelVerificationFailed)
			{
				System.Collections.ArrayList unitArray = new ArrayList();
				unitArray.Add(this.OriginalUnit);
				System.Data.DataTable dtOrderedUnit = BOL.OrderedUnit.GetOrderedUnitDataTableForReleaseOfPatientAssignment(unitArray, this.ModificationDateTime, this.ModificationTechId);

				dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtOrderedUnit, this._lastUpdateFunctionId));
				sprocArrayList.Add(SPROC.ReleaseOrderedUnitsByBloodUnit.StoredProcName);

				//Update the QuarantineIndicator
				System.Data.DataTable dtUpdateBUS = DAL.BloodUnitStatus.GetEmptyBloodUnitStatusTableSchema(true);

				System.Data.DataRow drBUS = this.ModifiedUnit.NewUnit.BloodUnitStatus.LoadDataRowFromThis(dtUpdateBUS.NewRow());
				drBUS[TABLE.BloodUnitStatus.UnitStatusDate] = this.ModificationDateTime;
				drBUS[TABLE.BloodUnitStatus.UnitStatusComments] = "Label Verification Failed";
				drBUS[TABLE.BloodUnitStatus.QuarantineIndicator] = true;
				dtUpdateBUS.Rows.Add(drBUS);

				dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtUpdateBUS, this._lastUpdateFunctionId));
				sprocArrayList.Add(SPROC.UpdateBloodUnitStatus.StoredProcName);
			}
			else
			{

				System.Data.DataTable dtUpdateOU = BOL.OrderedUnit.GetEmptyOrderedUnitDataTable();
				System.Data.DataTable dtAssignments = BOL.OrderedUnit.GetOrderedUnit(this.OriginalUnit.BloodUnitGuid, Common.LogonUser.LogonUserDivisionCode );

				foreach(System.Data.DataRow dr in dtAssignments.Rows)
				{
					bool createReleaseRecord = true;		//Flag to indicate if we need to release the assignment

					if (this._inheritedPatientAssignment[0] != System.Guid.Empty)
					{
						System.Guid unitGuid = this._inheritedPatientAssignment[0];
						System.Guid patientGuid = this._inheritedPatientAssignment[1];

						if ((System.Guid)dr[TABLE.BloodUnit.BloodUnitGuid] == unitGuid &&
							(System.Guid)dr[TABLE.Patient.PatientGuid] == patientGuid)
						{
							//This is the patient they want to carry over to the target, so we leave it alone
							createReleaseRecord = false;
						}
					}

					if (createReleaseRecord)
					{
						System.Data.DataRow drRelease = dtUpdateOU.NewRow();
						drRelease[TABLE.OrderedUnit.BloodUnitGuid] = this.OriginalUnit.BloodUnitGuid;
						drRelease[TABLE.OrderedUnit.ReleaseTechId] = this.ModificationTechId;
						drRelease[TABLE.OrderedUnit.ReleaseDate] = this.ModificationDateTime;
						drRelease[TABLE.OrderedUnit.LastUpdateUser] = Common.LogonUser.LogonUserName;
						drRelease[TABLE.OrderedUnit.RowVersion] = dr[TABLE.OrderedUnit.RowVersion];
						drRelease[TABLE.OrderedUnit.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
						dtUpdateOU.Rows.Add(drRelease);
					}
				}
				dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtUpdateOU, this._lastUpdateFunctionId));
				sprocArrayList.Add(SPROC.ReleaseOrderedUnitsByBloodUnit.StoredProcName);
			}


			//*	spInsertModificationSupply*************************************
			System.Data.DataTable dtModificationSupply = this.ModifiedUnit.LoadModificationSupplyInformation(bumGuids,false);
				
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtModificationSupply, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertModificationSupply.StoredProcName);

			//*	spInsertModificationEquipment**********************************
			System.Data.DataTable dtModificationEquipment = this.ModifiedUnit.LoadModificationEquipmentInformation(bumGuids,false);
				
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtModificationEquipment, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertModificationEquipment.StoredProcName);

			//Any exceptions generated by supplies  -- spInsertExceptionModification****************
			dtArrayList.Add(this.GetSupplyExceptionsDataTable(this.ModifiedUnit.Container));
			sprocArrayList.Add(SPROC.InsertExceptionUnitModification.StoredProcName);

			foreach(BOL.Supply tmpSupply in this.ModifiedUnit.SupplyList)
			{
				dtArrayList.Add(this.GetSupplyExceptionsDataTable(tmpSupply));
				sprocArrayList.Add(SPROC.InsertExceptionUnitModification.StoredProcName);
			}
			dtArrayList.Add(this.GetSupplyExceptionsDataTable(this.ModifiedUnit.SCDWafer));
			sprocArrayList.Add(SPROC.InsertExceptionUnitModification.StoredProcName);

			//*spInsertExceptionModification********************************************
			dtArrayList.Add(this.GetExceptionsDataTable(this.Exceptions));
			sprocArrayList.Add(SPROC.InsertExceptionUnitModification.StoredProcName);

			dtArrayList.Add(this.GetExceptionsDataTable(this.LabelExceptions));
			sprocArrayList.Add(SPROC.InsertExceptionLabelMismatch.StoredProcName);
			
			//* spInsertWorkloadEvents*****************************************
			System.Guid [] bloodUnitGuids = { this.ModifiedUnit.ToBloodUnitGuid };
			System.Data.DataTable dtWorkload = BOL.WorkloadEvent.GenerateWorkloadData(this.WorkloadCode, bloodUnitGuids, Common.WorkloadTransactionType.Unit, false);
			dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(dtWorkload, this._lastUpdateFunctionId));
			sprocArrayList.Add(SPROC.InsertWorkloadEvents.StoredProcName);

			DataTable dtBCEMessages = null;
			DataTable dtTempBCEMessages = null;

			//CR 2941
			//CR 2945
			//generate table with BCE_MESSAGE
			//getting all patients assigned to a unit
			//The Unit is modified, send BCE message releasing all unit assignments
			DataTable dtPatients = DAL.PatientAssignments.GetPatientsAssignmentDetails(this._originalUnit.BloodUnitGuid);
			Guid assignedPatientGuid = Guid.Empty;
			DataTable dtUnitAssignments = BOL.OrderedUnit.GetOrderedUnit(this._originalUnit.BloodUnitGuid, Common.LogonUser.LogonUserDivisionCode );
			for (int j=0; j< dtPatients.Rows.Count; j++)
			{
				assignedPatientGuid = (Guid)dtPatients.Rows[j][Common.VbecsTables.Patient.PatientGuid];

				//CR 3015
				if (dtBCEMessages == null)
				{
					dtBCEMessages = OrderedUnit.UpdateBceForRelease(this._originalUnit.BloodUnitGuid, assignedPatientGuid, false);
				}
				else
				{
					dtTempBCEMessages = OrderedUnit.UpdateBceForRelease(this._originalUnit.BloodUnitGuid, assignedPatientGuid, false);
					if (dtTempBCEMessages.Rows.Count > 0)
						dtBCEMessages.Rows.Add(dtTempBCEMessages.Rows[0].ItemArray);
				}
			}

			if (dtBCEMessages != null)
			{
				if (dtBCEMessages.Rows.Count > 0)
				{
					dtArrayList.Add(dtBCEMessages);
					sprocArrayList.Add(SPROC.HL7InsertMessageLog.StoredProcName);
				}
			}

			//BCE_MESSAGE end

			int result = (new Common.StoredProcedure().TransactionalGetValue(sprocArrayList, dtArrayList));

			//send BCE message reserving a new unit only if the user decided to move unit assignment during modification and 
			//if the label verification did not fail

			if (result == 0)
			{
				if (this.ModifiedUnit.LabelVerificationFailed == false)
				{
					if (this._inheritedPatientAssignment[0] != System.Guid.Empty)
					{
						System.Guid patientGuid = this._inheritedPatientAssignment[1];
						//CR 3015
						Guid orderedComponentGuid = Guid.Empty;
						DataRow[] drAssignments = dtUnitAssignments.Select(Common.VbecsTables.Patient.PatientGuid + " = '" + patientGuid.ToString() + "'");
						if (drAssignments.Length > 0)
							orderedComponentGuid = (Guid)drAssignments[0][Common.VbecsTables.OrderedUnit.OrderedComponentGuid];
						//CR 3057 Replaced call to UpdateBce with UpdateAndSaveBce
						OrderedUnit.UpdateAndSaveBce(this.ModifiedUnit.NewUnit.BloodUnitGuid, patientGuid, orderedComponentGuid, "", Common.HL7BPDispenseStatus.RS);
					}
				}
			}
			return result;
		}

		/// <summary>
		/// Gets the exceptions and returns a datatable with the needed information
		/// </summary>
		/// <param name="exceptions"></param>
		/// <returns></returns>
		private System.Data.DataTable GetExceptionsDataTable(System.Collections.ArrayList exceptions)
		{
			bool tableCloned = false;
			System.Data.DataTable dtExceptions = new System.Data.DataTable();
			
			foreach(BOL.ExceptionReport exRep in exceptions)
			{
				if (!tableCloned)
				{
					tableCloned = true;
					dtExceptions = exRep.ExceptionData.Table.Clone();
				}

				dtExceptions.Rows.Add(exRep.ExceptionData.ItemArray);
			}

			return(dtExceptions);
		}


		/// <summary>
		/// Gets any exceptions and returns a datatable with the needed information for a given Supply
		/// </summary>
		/// <param name="unitSupply"></param>
		/// <returns></returns>
		private System.Data.DataTable GetSupplyExceptionsDataTable(BOL.Supply unitSupply)
		{
			System.Data.DataTable dtExceptions = new System.Data.DataTable();

			if (unitSupply.LotException != null)
			{
				dtExceptions = unitSupply.LotException.ExceptionData.Table.Clone();
				dtExceptions.Rows.Add(unitSupply.LotException.ExceptionData.ItemArray);
			}

			return(dtExceptions);
		}

		/// <summary>
		/// Populates the input datarow with information specific to the BloodUnitModification table
		/// </summary>
		/// <param name="modifiedUnit">The 'target' unit will house all the data for the Modification table</param>
		/// <param name="dtRow"></param>
		/// <param name="originalUnit">Unmodified Unit</param>
		/// <returns>Datarow with BloodUnitModification Information</returns>
		private System.Data.DataRow LoadBloodUnitModificationInformation(BOL.ModifiedUnit modifiedUnit, System.Data.DataRow dtRow, BOL.BloodUnit originalUnit)
		{
			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitModification.BloodUnitModificationGuid))
				dtRow[TABLE.BloodUnitModification.BloodUnitModificationGuid] = modifiedUnit.BloodUnitModificationGuid;

			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitModification.DivisionCode))
				dtRow[TABLE.BloodUnitModification.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;

			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitModification.FromBloodUnitGuid))
				dtRow[TABLE.BloodUnitModification.FromBloodUnitGuid] = modifiedUnit.FromBloodUnitGuid;

			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitModification.ModificationDate))
				dtRow[TABLE.BloodUnitModification.ModificationDate] = modifiedUnit.ModificationDateTime;

			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitModification.ModificationMethodCode))
				dtRow[TABLE.BloodUnitModification.ModificationMethodCode] = Common.Utility.GetModificationMethodCharFromEnum(modifiedUnit.ModificationMethod);

			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitModification.ModificationTechId))
				dtRow[TABLE.BloodUnitModification.ModificationTechId] = this.ModificationTechId;

			//Only populate this field if we NEED to (SCD only) - so that the report can distinguish NULL from False
			if (this.ModificationMethod == Common.ModifyUnitMethod.SterileConnectingDevice)
			{
				if (dtRow.Table.Columns.Contains(TABLE.BloodUnitModification.NewContainerWeldIndicator))
					dtRow[TABLE.BloodUnitModification.NewContainerWeldIndicator] = this.NewContainerWeldIndicator;
			}

			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitModification.NewLabelVerificationDatetime))
			{
				if (modifiedUnit.NewLabelVerificationDateTime != System.DateTime.MinValue)
					dtRow[TABLE.BloodUnitModification.NewLabelVerificationDatetime] = modifiedUnit.NewLabelVerificationDateTime;
			}

			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitModification.ProductModificationCode))
				dtRow[TABLE.BloodUnitModification.ProductModificationCode] = Common.Utility.GetProductModificationCharFromEnum(modifiedUnit.ProductModificationCode);

			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitModification.ToBloodUnitGuid))
				dtRow[TABLE.BloodUnitModification.ToBloodUnitGuid] = modifiedUnit.ToBloodUnitGuid;

			
			
			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitModification.FromOriginalVolume))
				dtRow[TABLE.BloodUnitModification.FromOriginalVolume] = originalUnit.OriginalVolume;

			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitModification.FromBloodProductGuid))
				dtRow[TABLE.BloodUnitModification.FromBloodProductGuid] = originalUnit.BloodProductGuid;

			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitModification.FromUnitExpirationDate))
				dtRow[TABLE.BloodUnitModification.FromUnitExpirationDate] = originalUnit.BloodUnitMedia.UnitExpirationDate;

			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitModification.Comments))
			{
				if (this.Comments != string.Empty)
				{
					dtRow[TABLE.BloodUnitModification.Comments] = this.Comments;
				}
			}

			if (this.ProductModificationCode == Common.ProductModification.EditPool)
			{
				if (dtRow.Table.Columns.Contains(TABLE.BloodUnitModification.PoolEditedIndicator))
					dtRow[TABLE.BloodUnitModification.PoolEditedIndicator] = true;
			}

			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitModification.OriginalUnitIndicator))
				dtRow[TABLE.BloodUnitModification.OriginalUnitIndicator] = true;		//Set to false for New Inserts on EditPool later

			//We're going to store the modification cost based on the type performed -- 
			//For splits the modification cost will be a fraction of the action cost 
			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitModification.ModificationCost))
			{
				decimal childUnits = 1;

				if (this.IsSplit)
				{
					childUnits = (decimal)this.SplitCount;
				}
				
				//Calculate the cost for the actual Modification -- a defined in UC_9
				decimal childModificationCost = decimal.Divide(this.ModificationCost, childUnits);
				childModificationCost = decimal.Round(childModificationCost, 2);
				if (childModificationCost * childUnits < this.ModificationCost)
					childModificationCost += (decimal).01;

				dtRow[TABLE.BloodUnitModification.ModificationCost] = childModificationCost;
			}

			return dtRow;
		}

		/// <summary>
		/// Calculates the volumes/costs for the splitunit and it's original unit, stored in private variables
		/// Implements BR_25.05,BR_25.12, BR_25.15
		/// </summary>
		private void CalculateSplitProperties()
		{
			decimal childCost = decimal.Zero;
			decimal childSTCost = decimal.Zero;

			//Calculate the cost for the actual Modification -- as defined in UC_9
			decimal modificationCost = decimal.Divide(this.ModificationCost, (decimal)this.SplitCount);
			modificationCost = decimal.Round(modificationCost, 2);
			if (decimal.Multiply(modificationCost, (decimal)this.SplitCount) < this.ModificationCost)
				modificationCost += (decimal).01;

			int nCount = 0;
			//BR_25.01 - childCost is a fraction of the original unit's cost 
			if (this.SplittingUnit.BloodUnitFinance.UnitCost != System.Decimal.MinValue &&
				this.SplittingUnit.BloodUnitFinance.UnitCost != decimal.Zero)
			{
				childCost = decimal.Divide(this.SplittingUnit.BloodUnitFinance.UnitCost, (decimal)this.SplitCount);
				childCost = decimal.Round(childCost, 2);
				if (decimal.Multiply(childCost, (decimal)this.SplitCount) < this.SplittingUnit.BloodUnitFinance.UnitCost)
					childCost += (decimal).01;		//If there's anything leftover, add a penny
			}

			//The Special Testing Cost should be split amongst the children as well
			if (this.SplittingUnit.BloodUnitFinance.SpecialTestingCost != System.Decimal.MinValue &&
				this.SplittingUnit.BloodUnitFinance.SpecialTestingCost != decimal.Zero)
			{
				childSTCost = decimal.Divide(this.SplittingUnit.BloodUnitFinance.SpecialTestingCost, (decimal)this.SplitCount);
				childSTCost = decimal.Round(childSTCost, 2);
				if (decimal.Multiply(childSTCost, (decimal)this.SplitCount) < this.SplittingUnit.BloodUnitFinance.UnitCost)
					childSTCost += (decimal).01;		//If there's anything leftover, add a penny
			}

			//BR_25.05 - childVolume is a fraction of the original unit volume
			//BR_25.05 - modVolume is the remainder volume.  10/3 has remainder 1.  This is addeded below
			decimal modVolume = this.SplittingUnit.OriginalVolume % this.SplitCount;
			decimal childVolume = (this.SplittingUnit.OriginalVolume - modVolume) / this.SplitCount;
			
			//Perhaps not the best day to do this, but it's all i've come up with for now
			//maybe setup an event that could fire off an update the volume???
			//BR_25.15 BR_25.12 - the child unit's volume is calculated
			nCount = this._splitUnits.Count;
			for(int ndx=0; ndx < nCount; ndx++)
			{
				BOL.ModifiedUnit child = (BOL.ModifiedUnit)this._splitUnits[ndx];
				//According to      , make one unit have more, rather than give the leftover to original
				if (ndx == 0)
					child.NewUnit.OriginalVolume = childVolume + modVolume;
				else
					child.NewUnit.OriginalVolume = childVolume;
				child.NewUnit.BloodUnitFinance.UnitCost = childCost + modificationCost;
				child.NewUnit.BloodUnitFinance.SpecialTestingCost = childSTCost;
			}

			if (this.ModificationMethod == Common.ModifyUnitMethod.Closed)
				this.SetOriginalContainerIndicator(1, true);
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/25/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6711"> 
		///		<ExpectedInput>Valid BloodUnit</ExpectedInput>
		///		<ExpectedOutput>Arraylist</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6712"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Checks through business rules to make sure a unit is ok before adding.
		///If we find something REALLY bad, we set the Cancel Action and just get out (return).
		///Implements BR_25.21,BR_25.01,BR_26.32,BR_26.08,BR_26.05,BR_26.06,BR_26.09,BR_26.03,BR_26.04,BR_26.42,BR_26.07
		/// </summary>
		/// <param name="testBloodUnit">Unit we're going to validate</param>
		/// <param name="loggedUser">Current user - for checking security</param>
		/// <returns></returns>
		public System.Collections.ArrayList ValidateBusinessRules(BOL.BloodUnit testBloodUnit, BOL.VbecsUser loggedUser)
		{
			System.Collections.ArrayList tmpList = new System.Collections.ArrayList();

			//Save off our original expiration date
			System.DateTime expireCheck = testBloodUnit.BloodUnitMedia.UnitExpirationDate;

			//This is according to the Limitations and Restrictions section of UC_24
			if (this.ProductModificationCode == Common.ProductModification.EditPool)
			{
				if (testBloodUnit.PatientAssignments.Rows.Count > 0)
				{
					if (!loggedUser.IsAuthorized(Common.UserRoleType.TraditionalSupervisor))
					{
						tmpList.Add(new Common.ActionItem(Common.VerifyAction.Cancel, Common.StrRes.InfoMsg.Common.InsufficientSecurityPrivileges("edit this pooled unit").ResString));
						return(tmpList);
					}
				}
			}

			if (this.ProductModificationCode == Common.ProductModification.VolumeReduced)
			{
				if (testBloodUnit.OriginalVolume < (decimal)2)
				{
					//"Unit volume is too low.  Cannot modify."
					tmpList.Add(new Common.ActionItem(Common.VerifyAction.Cancel, Common.StrRes.InfoMsg.UC026.VolumeToLow().ResString));
					return(tmpList);
				}
			}

			if (this.IsSplit)
			{	
				//if the sevcnth char is a number or little letter say not allowed.
				if(testBloodUnit.UnitDivisionId.Trim() != "00")
				{
					if(char.IsNumber(testBloodUnit.UnitDivisionId.Substring(0,1), 0) || char.IsLower(testBloodUnit.UnitDivisionId.Substring(0,1), 0)) 
					{
						// "Split is not allowed. The product code format is not supported for split."
						tmpList.Add(new Common.ActionItem(Common.VerifyAction.Cancel, Common.StrRes.InfoMsg.UC025.SplitNotAllowed().ResString));
						return(tmpList);
					
					}
				}
				
				//BR_25.21 -- Check to see if the unit's already been split or pooled
				if (BOL.BloodUnitModification.HasUnitBeenSplit(testBloodUnit) || testBloodUnit.PooledUnitsCount > 0 )
				{
					tmpList.Add(new Common.ActionItem(Common.VerifyAction.Cancel, Common.StrRes.SysErrMsg.UC025.UnitAlreadySplit().ResString));
					return(tmpList);
				}

				//BR_25.01 -- The Original Volume must be great enough to split into 2 parts
				if (testBloodUnit.OriginalVolume < (decimal)2)
				{
					tmpList.Add(new Common.ActionItem(Common.VerifyAction.Cancel, Common.StrRes.SysErrMsg.UC025.NotEnoughVolumeToSplit().ResString));
					return(tmpList);
				}
			}

			//BR_26.33 - Just in case the sproc is wrong, this BR will pickup the slack
			if (testBloodUnit.BloodUnitStatus.UnitStatusCode != Common.UnitStatusCode.Available &&
				testBloodUnit.BloodUnitStatus.UnitStatusCode != Common.UnitStatusCode.Assigned &&
				testBloodUnit.BloodUnitStatus.UnitStatusCode != Common.UnitStatusCode.CrossmatchedAssigned &&
				testBloodUnit.BloodUnitStatus.UnitStatusCode != Common.UnitStatusCode.Limited)
			{
				tmpList.Add(new Common.ActionItem(Common.VerifyAction.Cancel, Common.StrRes.ValidMsg.UC026.BR33InvalidUnitStatus(testBloodUnit.EyeReadableUnitId, testBloodUnit.ProductType.ProductTypeText, testBloodUnit.BloodUnitStatus.UnitStatusText).ResString));
				return(tmpList);
			}

			//BR_26.08 - Modification date cannot be earlier than the unit's received date
			if (testBloodUnit.IncomingShipment.ShipmentDate > this.ModificationDateTime)
			{
				tmpList.Add(new Common.ActionItem(Common.VerifyAction.Cancel, Common.StrRes.ValidMsg.UC026.BR8ModificationDate().ResString));
				return(tmpList);
			}
			

			//BR_26.05 & BR_26.06
			if (testBloodUnit.BloodUnitStatus.QuarantineIndicator)
			{
				if (loggedUser.IsAuthorized(Common.UserRoleType.EnhancedTechnologist))
				{
					tmpList.Add(new Common.ActionItem(Common.VerifyAction.Release, Common.StrRes.ValidMsg.UC026.BR6QuarantineUnit(testBloodUnit.EyeReadableUnitId,testBloodUnit.ProductType.ProductTypeText).ResString));
				}
				else
				{
					tmpList.Add(new Common.ActionItem(Common.VerifyAction.Cancel, Common.StrRes.ValidMsg.UC026.BR5QuarantineUnit(testBloodUnit.EyeReadableUnitId,testBloodUnit.ProductType.ProductTypeText).ResString));
					return(tmpList);
				}
			}

			//BR_26.09 -- needs to apply before BR_26.03/BR_26.04 are checked
			if ((testBloodUnit.ContainsRedBloodCells()) && (this.ProductModificationCode == Common.ProductModification.Rejuvenated))
			{
				expireCheck = expireCheck.Date.AddDays(3).AddHours(23).AddMinutes(59).AddSeconds(59);
			}

			//BR_26.03 & BR_26.04
			if (this.ModificationDateTime > expireCheck)
			{
				if (loggedUser.IsAuthorized(Common.UserRoleType.EnhancedTechnologist))
				{
					tmpList.Add(new Common.ActionItem(Common.VerifyAction.OverrideExpiration, Common.StrRes.ValidMsg.UC026.BR4ExpiredUnit(testBloodUnit.EyeReadableUnitId,testBloodUnit.ProductType.ProductTypeText).ResString));
				}
				else
				{
					tmpList.Clear();
					tmpList.Add(new Common.ActionItem(Common.VerifyAction.Cancel, Common.StrRes.ValidMsg.UC026.BR3ExpiredUnit(testBloodUnit.EyeReadableUnitId,testBloodUnit.ProductType.ProductTypeText).ResString));
					return(tmpList);
				}
			}

			//BR_26.42
			if (testBloodUnit.BloodUnitStatus.BiohazardousWarningIndicator)
			{
				tmpList.Add(new Common.ActionItem(Common.VerifyAction.OverrideBiohazard, Common.StrRes.ValidMsg.UC026.BR42Biohazard(testBloodUnit.EyeReadableUnitId,testBloodUnit.ProductType.ProductTypeText).ResString));
			}

			//BR_26.07 - ABORh testing is required -- updated check due to CR3558
			if (testBloodUnit.ProductType.AboRhConfirmationRequired && testBloodUnit.BloodUnitStatus.UnitStatusCode == Common.UnitStatusCode.Limited)
			{
				tmpList.Add(new Common.ActionItem(Common.VerifyAction.Warning, Common.StrRes.ValidMsg.UC026.BR7ABORh(testBloodUnit.EyeReadableUnitId, testBloodUnit.ProductType.ProductTypeText).ResString));
			}

			return(tmpList);
		}


		/// <summary>
		/// Validates the incoming bloodunit with our first unit in pooling mass -- BRs
		/// Implements BR_24.20, BR_24.19,BR_24.28 (BR_24.15), BR_24.02 + BR_24.28
		/// BR_24.16, BR_24.17, BR_24.20
		/// </summary>
		/// <param name="newUnit"></param>
		/// <returns></returns>
		private Common.ActionItem ValidatePoolRules(BOL.BloodUnit newUnit)
		{
			System.Collections.ArrayList actionItems = new System.Collections.ArrayList();
			BOL.BloodUnit existingUnit = (BOL.BloodUnit)this._poolingUnits[0];

			bool newUnitIsAutologous = 	BOL.BloodUnit.IsAutologous(newUnit.DonationType.DonationTypeCode);
			bool newUnitIsDirected = BOL.BloodUnit.IsDirected(newUnit.DonationType.DonationTypeCode);
			bool newUnitIsAllogenic = BOL.BloodUnit.IsAllogenic(newUnit.DonationType.DonationTypeCode);

			//BR_24.02
			if (existingUnit.BloodComponentClassId != newUnit.BloodComponentClassId)
			{
				return new Common.ActionItem(Common.VerifyAction.Cancel, Common.StrRes.ValidMsg.UC024.BR02ComponentType(newUnit.EyeReadableUnitId, newUnit.ProductType.ProductTypeText, existingUnit.BloodComponentName).ResString);
			}

			//BR_24.15
			if (existingUnit.DonationType.DirectedType != newUnit.DonationType.DirectedType)
			{
				string directedTypeText = "Allogeneic";
				if (existingUnit.DonationType.DirectedType == "D")	directedTypeText = "Directed";
				if (existingUnit.DonationType.DirectedType == "A")	directedTypeText = "Autologous";

				return new Common.ActionItem(Common.VerifyAction.Cancel, Common.StrRes.ValidMsg.UC024.BR15DonationType(newUnit.EyeReadableUnitId, newUnit.ProductType.ProductTypeText, directedTypeText).ResString);
			}

			//BR_24.16  - The Restricted for patients MUST be the same, if none -> the guid = Empty
			if (existingUnit.BloodUnitMedia.DirectedPatientGuid != newUnit.BloodUnitMedia.DirectedPatientGuid)
			{
				return new Common.ActionItem(Common.VerifyAction.Cancel, Common.StrRes.ValidMsg.UC024.BR16DifferentPatient(newUnit.EyeReadableUnitId, newUnit.ProductType.ProductTypeText).ResString);
			}
			
			//BR_24.17  -- should I check for patient assignment if both are Autologous?
			if (newUnitIsAutologous && (BOL.BloodUnit.IsDirected(existingUnit.DonationType.DonationTypeCode) || BOL.BloodUnit.IsAllogenic(existingUnit.DonationType.DonationTypeCode)))
			{
				return new Common.ActionItem(Common.VerifyAction.Cancel, Common.StrRes.ValidMsg.UC024.BR17Autologous(newUnit.EyeReadableUnitId, newUnit.ProductType.ProductTypeText).ResString);
			}

			//Editing a pool has special rules that apply - Cancels are all done, so we can do override checks now
			if (this.ProductModificationCode == Common.ProductModification.EditPool)
			{
				//BR_24.28 - Since only one Action is being returned, let the cancel's hit first   //producttype is covered in ValidateAddingPooledUnit
				if (newUnit.DonationType.DirectedType != this.PooledUnit.NewUnit.DonationType.DirectedType)
				{
					return new Common.ActionItem(Common.VerifyAction.Cancel, Common.StrRes.InfoMsg.Common.Dynamic("You can only add new units of the same product type and donation type as the pooled unit.").ResString);
				}

				//BR_24.25 -- Check the expiration date of the new unit compared to the existing pool
				if (newUnit.BloodUnitMedia.UnitExpirationDate < this.PooledUnit.UnitExpirationDate)
				{
					return new Common.ActionItem(Common.VerifyAction.OverrideExpiration, Common.StrRes.ValidMsg.UC024.BR25ExpirationDate().ResString);
				}
			}

			return new Common.ActionItem(Common.VerifyAction.Continue,string.Empty);
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/4/2006</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8426"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>True</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8427"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Flase</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns true if the Modification is any POOL type
		/// </summary>
		public bool IsPool
		{
			get
			{
				return(this.ProductModificationCode == Common.ProductModification.EditPool ||
					this.ProductModificationCode == Common.ProductModification.Pooled ||
					this.ProductModificationCode == Common.ProductModification.PoolThawCryo);
			}
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/4/2006</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8428"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>True</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8429"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>False</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns true if the BloodUnitModification is a Split
		/// </summary>
		public bool IsSplit
		{
			get
			{
				return(this.ProductModificationCode == Common.ProductModification.SplitAdult);
			}
		}

		/// <summary>
		/// Cycles through all the pooling units and calculates stuff
		/// Implements BR_24.13, BR_24.08, BR_24.03, BR_24.13, BR_24.09 
		/// </summary>
		private void CalculatePooledProperties()
		{
			bool mixedABO = false;
			bool mixedRH = false;
			decimal unitCost = decimal.Zero;
			decimal testingCost = decimal.Zero;
			decimal unitVolume = decimal.Zero;

			Common.ABO tempAbo = Common.ABO.NA;
			Common.RH tempRh = Common.RH.NotProvided;

			bool firstPass = true;

			//Go through all our input units and calculate the totals
			foreach(BOL.BloodUnit testUnit in this._poolingUnits)
			{			
				if (firstPass)
				{
					tempAbo = testUnit.BloodUnitMedia.AboRh.Abo;
					tempRh = testUnit.BloodUnitMedia.AboRh.RH;
					firstPass = false;
				}

				if (testUnit.BloodUnitMedia.AboRh.Abo != tempAbo)
					mixedABO = true;
				
				if (testUnit.BloodUnitMedia.AboRh.RH != tempRh)
					mixedRH = true;

				//BR_24.08 - sum the volumes
				unitVolume += testUnit.OriginalVolume;

				//BR_24.03 - sum the unitcosts
				if (testUnit.BloodUnitFinance.UnitCost != decimal.MinValue)
					unitCost += testUnit.BloodUnitFinance.UnitCost;

				if (testUnit.BloodUnitFinance.SpecialTestingCost != decimal.MinValue)
					testingCost += testUnit.BloodUnitFinance.SpecialTestingCost;
			}

			//BR_24.13 - assignment
			this._pooledUnit.UnitExpirationDate = this._pooledUnit.RecommendedExpirationDateTime(this);

			//BR_24.09 -- After cylcing through all pooling units, assign the ABORh values
			if (mixedABO)
				this._pooledUnit.NewUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			else
				this._pooledUnit.NewUnit.BloodUnitMedia.AboRh.Abo = tempAbo;

			if (mixedRH)
				this._pooledUnit.NewUnit.BloodUnitMedia.AboRh.RH = Common.RH.PooledRH;
			else
				this._pooledUnit.NewUnit.BloodUnitMedia.AboRh.RH = tempRh;


			this._pooledUnit.NewUnit.OriginalVolume = unitVolume;
			this._pooledUnit.NewUnit.BloodUnitFinance.UnitCost = unitCost + this.ModificationCost;
			this._pooledUnit.NewUnit.BloodUnitFinance.SpecialTestingCost = testingCost;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7020"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Comma Delimited String</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7021"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns an array of comma-delimated bloodproduct guids of the original unit(s)
		/// Needed for pooling, since each pooling guid determines the next available target
		/// </summary>
		/// <returns></returns>
		public string GetBloodProductGuidArray()
		{
			string guidArray = string.Empty;
			string strComma = string.Empty;

			if (this.IsPool)
			{
				foreach(BOL.BloodUnit tmpUnit in this._poolingUnits)
				{
					guidArray += strComma + tmpUnit.BloodProductGuid.ToString();
					strComma = ",";
				}
			}
			else if(this.IsSplit)
			{
				guidArray = this._splittingUnit.BloodProductGuid.ToString();
			}
			else
			{
				guidArray = this._originalUnit.BloodProductGuid.ToString();
			}

			return(guidArray);
		}

		/// <summary>
		/// Returns a datatable of original unit's antigen information for target save
		/// </summary>
		/// <param name="dsOriginalUnitAntigens"></param>
		/// <param name="targetUnit"></param>
		/// <returns></returns>
		private System.Data.DataTable LoadAntigenInformation(System.Data.DataSet dsOriginalUnitAntigens, BOL.ModifiedUnit targetUnit)
		{
			System.Data.DataTable dtBloodUnitAntigens = BOL.BloodUnitAntigen.GetEmptyBloodUnitAntigenTable(false);

			if (dsOriginalUnitAntigens.Tables.Count > 1)
			{
				foreach(System.Data.DataRow dr in dsOriginalUnitAntigens.Tables[1].Rows)
				{
					BOL.BloodUnitAntigen antigen = new BOL.BloodUnitAntigen(dr);
					antigen.BloodUnitGuid = targetUnit.ToBloodUnitGuid;
					antigen.RecordStatusCode = Common.RecordStatusCode.Active;
					dtBloodUnitAntigens.Rows.Add(this.LoadBloodUnitAntigenInformation(antigen, dtBloodUnitAntigens.NewRow()));
				}

			}
			return(dtBloodUnitAntigens);
		}

		/// <summary>
		/// Returns a datatable of original unit's relative test information for target save
		/// Implements BR_26.23
		/// </summary>
		/// <param name="dtOriginalTests"></param>
		/// <param name="targetBloodUnitGuid"></param>
		/// <returns></returns>
		private System.Data.DataTable LoadBloodUnitTestInformation(System.Data.DataTable dtOriginalTests, System.Guid targetBloodUnitGuid)
		{
			System.Data.DataTable dtBloodUnitTest = DAL.BloodUnitTest.GetEmptyBloodUnitTestTableSchema(false);
			BOL.BloodUnitTest addingTest;
			int unitTestId = int.MinValue;

			//No rows were found --- insert anything?
			if (dtOriginalTests.Rows.Count > 0)
			{
				//Loop through all the original units tests out there
				foreach(System.Data.DataRow dr in dtOriginalTests.Rows)
				{
					unitTestId = (int)dr[TABLE.BloodUnitTest.BloodTestTypeId];

					//BR_26.23 -- new units get original unit's AboRh confirmation testing
					if (unitTestId == (int)Common.TestType.ABOInterp ||
						unitTestId == (int)Common.TestType.RhInterp ||
						unitTestId == (int)Common.TestType.SickleCell ||
						unitTestId == (int)Common.TestType.CMV)
					{
						//Create a 'blank' BloodUnitTest for display purposes only
						//RecordStatusCode is set to 'A' in spInsertBloodUnitTest
						addingTest = new BOL.BloodUnitTest(targetBloodUnitGuid, unitTestId);
						addingTest.TestDate = this.ModificationDateTime;
						addingTest.TestResultId = dr[TABLE.BloodUnitTest.TestResultId].ToString();
						addingTest.EntryMethodCode = dr[TABLE.BloodUnitTest.EntryMethodCode].ToString();
						addingTest.RecordStatusCode = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Modified).ToString();
						dtBloodUnitTest.Rows.Add(addingTest.LoadDataRowFromThis(dtBloodUnitTest.NewRow()));
					}
				}
			}
			return(dtBloodUnitTest);

		}

		#endregion

	}
}
